<?php
/**
 * Modules core file.
 *
 * @package wp-travel-import-export-core
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Custom Filters Core.
if ( ! class_exists( 'WP_Travel_Authorize_Net_Checkout_Core' ) ) :
	/**
	 * WP Travel Custom Filters Core.
	 */
	class WP_Travel_Authorize_Net_Checkout_Core {

		const WP_TRAVEL_AUTHORIZE_NET_CHECKOUT_HANDLE = 'wp_travel_authorize_net_checkout_';

		/**
		 * Absolute path to the core module.
		 *
		 * @var string
		 */
		protected static $abspath;

		/**
		 * Plugin File.
		 *
		 * @var string
		 */
		protected static $plugin_file;

		/**
		 * The Addon Version.
		 *
		 * @var string
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP_Travel_Authorize_Net_Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Authorize_Net_Core Instance.
		 * Ensures only one instance of WP_Travel_Authorize_Net_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Authorize_Net_Core()
		 * @return WP_Travel_Authorize_Net_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $params Core class init paramerters.
		 */
		public static function init( $params ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$version     = $params['version'];
			self::includes();
			add_action( 'init', array( __CLASS__, 'init_hooks' ) );
		}

		/**
		 * Includes Essential Files.
		 *
		 * @return void
		 */
		public static function includes() {
			include_once self::$abspath . 'inc/functions.php';
		}

		/**
		 * Initialize Hooks.
		 */
		public static function init_hooks() {
			add_action( 'wp_enqueue_scripts', array( __CLASS__, 'assets_views' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'assets' ) );
			add_filter( 'wp_travel_display_general_admin_notices', array( __CLASS__, 'display_admin_general_notices' ) );
			add_action( 'wp_travel_general_admin_notice', array( __CLASS__, 'admin_general_notices' ) );
			add_filter( 'wp_travel_payment_gateway_lists', array( __CLASS__, 'register' ) );
			// add_filter( 'wp_travel_premium_addons_list', array( __CLASS__, 'add_to_premium_addons' ) );
			add_action( 'wp_footer', array( __CLASS__, 'add_iframe_container' ) );
			if ( is_admin() ) {
				// add_action( 'wp_travel_payment_gateway_fields', array( __CLASS__, 'gateway_setting_fields' ), 12, 2 );
				add_action( 'wp_travel_payment_gateway_fields_authorizenet', array( __CLASS__, 'gateway_setting_fields' ), 12, 2 );
				add_filter( 'wp_travel_block_before_save_settings', array( __CLASS__, 'save_settings_v4' ), 12, 2 );
				add_filter( 'wp_travel_before_save_settings', array( __CLASS__, 'save_settings' ) );
			}
			if ( self::is_authorizenet() ) {
				add_action( 'wp_travel_after_frontend_booking_save', array( __CLASS__, 'set_up_payment_options' ) );
			}
			add_action( 'wp_travel_before_partial_payment_complete', array( __CLASS__, 'set_up_payment_options' ), 10, 2 );
			$has_used_authorizenet = isset( $_SESSION['used-authorizenet-recently'] ) && $_SESSION['used-authorizenet-recently'];
			if ( $has_used_authorizenet ) {
				add_filter( 'wp_travel_booked_message', array( __CLASS__, 'booking_message' ) );
			}
		}

		/**
		 * Register/Enqueue Scripts.
		 *
		 * @return void
		 */
		public static function assets_views() {
			if ( function_exists( 'wptravel_can_load_payment_scripts' ) ) {
				$load_payment_script = wptravel_can_load_payment_scripts();
			} else {
				$load_payment_script = wp_travel_can_load_payment_scripts();
			}
			$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : 'min.';
			if ( $load_payment_script && self::is_enabled() ) {
				// $min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : 'min.';
				wp_enqueue_style( 'wp-travel-authorize-net-checkout-views', plugin_dir_url( __FILE__ ) . 'assets/css/wp-travel-authorize-net-checkout-views.' . $min . 'css', '', '1.0.0' );
				wp_register_script( 'wp-travel-authorize-net-checkout-views', plugin_dir_url( __FILE__ ) . 'assets/js/wp-travel-authorize-net-checkout-views.' . $min . 'js', array( 'jquery' ), '1.0.0', true );
				self::localize_script();
				wp_enqueue_script( 'wp-travel-authorize-net-checkout-views' );
			}
			global $post;
			// wp-travel-authorize-net-checkout-frontend
			if ( get_post_type() == 'itineraries' ) {
				$devp = array( 'wp-travel-frontend-booking-widget' );
				wp_enqueue_style( 'wp-travel-authorize-net-checkout-views', plugin_dir_url( __FILE__ ) . 'assets/css/wp-travel-authorize-net-checkout-views.' . $min . 'css', '', '1.0.0' );
				wp_register_script( 'wp-travel-authorize-net-checkout-frontend', plugin_dir_url( __FILE__ ) . 'app/build/wp-travel-booking.js', $devp, '1.0.0', true );
				self::localize_script_forntend();
				wp_enqueue_script( 'wp-travel-authorize-net-checkout-frontend' );
			}

		}

		/**
		 * Settings for the Plugin.
		 *
		 * @param array $args Settings array.
		 * @return void
		 */
		public static function gateway_setting_fields( $args ) {
			wp_enqueue_script( 'wp-travel-authorize-net-checkout' );
			// require 'inc/settings.php';
		}

		/**
		 * Register/Enqueue Scripts.
		 *
		 * @return void
		 */
		public static function assets() {
			$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : 'min.';

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			// settings_screen.
			$screen = get_current_screen();

			if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_AUTHORIZE_NET_CHECKOUT_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
			// wp_enqueue_style( 'wp-travel-authorize-net-checkout', plugin_dir_url( __FILE__ ) . 'assets/css/wp-travel-authorize-net-checkout.' . $min . 'css', '', '1.0.0' );
			// wp_register_script( 'wp-travel-authorize-net-checkout', plugin_dir_url( __FILE__ ) . 'assets/js/wp-travel-authorize-net-checkout.' . $min . 'js', array( 'jquery' ), '1.0.1', true );
			self::localize_script();
		}

		/**
		 * Push the new payment gateway on the payment gatewyas list.
		 *
		 * @param array $payments_gateways Array of payment gateways.
		 * @return array New gateway pushed Array.
		 */
		public static function register( $payments_gateways ) {
			$payments_gateways['authorizenet'] = 'Authorize.Net';
			return $payments_gateways;
		}

		/**
		 * Push the settings for the plugin to be saved.
		 *
		 * @param array $settings Settings array.
		 * @param array $settings_data Settings data array.
		 * @return array Array with new settings.
		 */
		public static function save_settings_v4( $settings, $settings_data ) {

			if ( ! $settings ) {
				return $settings;
			}
			$fields = array(
				'payment_option_authorizenet',
				'wp_travel_authorizenet_api_login_id',
				'wp_travel_authorizenet_transaction_key',
				'wp_travel_authorizenet_md5_hash_key',
				'wp_travel_authorizenet_valid_currency',
			);

			foreach ( $fields as $field ) {
				if ( isset( $settings_data[ $field ] ) ) {
					$settings[ $field ] = $settings_data[ $field ];
				}
			}
			if ( isset( $settings_data['wp_travel_authorizenet_valid_currency'] ) && 'yes' !== $settings_data['wp_travel_authorizenet_valid_currency'] ) {
				$settings['payment_option_authorizenet'] = 'no'; // Disable if not validated.
			}

			return $settings;
		}

		/**
		 * Push the settings for the plugin to be saved.
		 *
		 * @param array $settings Settings array.
		 * @return array Array with new settings.
		 */
		public static function save_settings( $settings ) {

			if ( ! $settings ) {
				return $settings;
			}
			$fields = array(
				'payment_option_authorizenet',
				'wp_travel_authorizenet_api_login_id',
				'wp_travel_authorizenet_transaction_key',
				'wp_travel_authorizenet_md5_hash_key',
				'wp_travel_authorizenet_valid_currency',
			);

			if ( isset( $_POST['wp_travel_authorizenet_valid_currency'] ) && 'yes' !== $_POST['wp_travel_authorizenet_valid_currency'] ) {
				unset( $fields[0] );
			}

			foreach ( $fields as $field ) {
				if ( isset( $_POST[ $field ] ) ) {
					$settings[ $field ] = $_POST[ $field ];
				}
			}

			return $settings;
		}


		/**
		 * Setup payment options.
		 *
		 * @param number $booking_id Booking ID.
		 * @return void
		 */
		public static function set_up_payment_options( $booking_id, $complete_partial_payment = false ) {

			if ( $complete_partial_payment && 'authorizenet' !== $_POST['wp_travel_payment_gateway'] ) {
				return;
			} elseif ( self::is_disabled() || ! self::is_authorizenet() || ! $booking_id ) {
				return;
			}

			$_SESSION['used-authorizenet-recently'] = true;

			do_action( 'wt_before_payment_process', $booking_id );

			$payment_id = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
			update_post_meta( $booking_id, 'wp_travel_booking_status', 'booked' );

			// get previous payment ids .
			$authorizenet_args = get_post_meta( $payment_id, '_authorizenet_args', true );

			// Payment Details.
			$json    = sanitize_text_field( wp_unslash( $_REQUEST['payment_details'] ) );
			$payment = json_decode( json_decode( $json ) );
			$amount  = $payment->{'totalAmount'};

			$payment_gateway = 'authorizenet';

			if ( '' !== $authorizenet_args ) { // Partial Payment.

				/**
				 * This will return the current payment id to insert payment meta for the current payment.
				 *
				 * @hooked wptravel_update_payment_post from partial payment which will also update booking option [booking_only | booking_with_payment ]
				 *
				 * @since 4.0.2
				 */
				$new_payment_id = apply_filters( 'wptravel_before_insert_partial_payment', $payment_id, $booking_id, $payment_gateway );
				$detail         = $payment;
				wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_gateway ), $new_payment_id );
			} else {
				update_post_meta( $payment_id, '_authorizenet_args', $payment );
				update_post_meta( $payment_id, 'wp_travel_payment_status', 'paid' );
				update_post_meta( $payment_id, 'wp_travel_payment_amount', $amount );

				update_post_meta( $booking_id, 'wp_travel_payment_status', 'paid' );
				do_action( 'wp_travel_after_successful_payment', $booking_id );
			}
			// wptravel_update_payment_status( $booking_id, $amount, 'paid', $payment, '_authorizenet_args' );
		}

		/**
		 * Checks if authorizenet is used as payment gateway.
		 *
		 * @return boolean
		 */
		private static function is_authorizenet() {
			$settings       = wptravel_get_settings();
			if( isset( $settings['selected_booking_option'] ) &&  count( $settings['selected_booking_option'] ) == 1 && $settings['selected_booking_option'][0] = 'booking-with-payment' ){
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'authorizenet' === $_REQUEST['wp_travel_payment_gateway'];
			}else{
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_booking_option', $_REQUEST ) && 'booking_with_payment' === $_REQUEST['wp_travel_booking_option'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'authorizenet' === $_REQUEST['wp_travel_payment_gateway'];
			}
			
		}

		/**
		 * Show / Hide admin notices.
		 *
		 * @return void
		 */
		public static function display_admin_general_notices( $show_notices ) {
			if ( ! is_ssl() && ! $show_notices ) :
				$show_notices = true;
			endif;
			return $show_notices;
		}

		/**
		 * Display Admin Notices.
		 *
		 * @return void
		 */
		public static function admin_general_notices() {
			if ( ! is_ssl() ) :
				?>
				<li>
					<p><?php esc_html_e( 'Authorize.Net: Authorize.Net recommends the secure connection to make transaction secure. Some browsers may block the unsafe scripts which may affects the plugin functionality.', 'wp-travel-pro' ); ?></p>
				</li>
				<?php
			endif;
		}

		/**
		 * Checks if the payment method is enabled,
		 *
		 * @return boolean
		 */
		private static function is_enabled() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			return array_key_exists( 'payment_option_authorizenet', $settings ) && 'yes' === $settings['payment_option_authorizenet'];
		}

		/**
		 * Determing is express checkout is disabled.
		 *
		 * @return boolean
		 */
		private static function is_disabled() {
			return ! self::is_enabled();
		}

		/**
		 * Sets the WP Travel Booking Message.
		 *
		 * @param string $message Booking Message.
		 * @return string New Booking Message.
		 */
		public static function booking_message( $message ) {
			unset( $_SESSION['used-authorizenet-recently'] );

			$message = esc_html__( "We've received your booking and payment details. We'll contact you soon.", 'wp-travel-pro' );

			return $message;
		}

		/**
		 * Adds html for iframe popup.
		 */
		public static function add_iframe_container() {
			if ( self::is_disabled() ) {
				return;
			}

			$include_file     = false;
			$setting          = wptravel_get_settings();
			$authorize_enable = isset( $setting['payment_option_authorizenet'] ) ? $setting['payment_option_authorizenet'] : false;
			if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
				$include_file = WP_Travel::is_page( 'dashboard' ) || WP_Travel::is_page( 'checkout' );
			} else {
				$include_file = wp_travel_is_dashboard_page() || wp_travel_is_checkout_page();
			}
			if ( $include_file == false && $authorize_enable == true ) {
				$include_file = true;
			}
			if ( $include_file ) {
				$iframe_src = plugin_dir_url( __FILE__ ) . 'templates/iframe.html';
				$protocol   = is_ssl() ? 'https' : 'http';
				$api_url    = "{$protocol}://test.authorize.net/payment/payment";

				if ( function_exists( 'wptravel_test_mode' ) ) {
					$test_mode = wptravel_test_mode();
				} else {
					$test_mode = wp_travel_test_mode();
				}
				if ( ! $test_mode ) {
					$api_url = "{$protocol}://accept.authorize.net/payment/payment";
				}
				include_once plugin_dir_path( __FILE__ ) . 'templates/iframe-container.php';
			}
			?>
			<?php
		}

		/**
		 * Localizes a 'wp-travel-authorizenet-script' script with data for a JavaScript variable.
		 */
		public static function localize_script() {
			$authorizenet_vars = array();
			if ( is_admin() ) {
				$authorizenet_vars['ajaxUrl']  = admin_url( 'admin-ajax.php' );
				$authorizenet_vars['messages'] = array(
					'account_validation_failed'      => __( 'Failed to Validate Account.', 'wp-travel-pro' ),
					'currency_acceptance'            => __( 'The Merchant account doesn\'t accepts the current store currency. The merchant account accepts following currency: ', 'wp-travel-pro' ),
					'currency_validation_successful' => __( 'Currency Validated Successfully.', 'wp-travel-pro' ),
				);
				wp_localize_script( 'wp-travel-authorize-net-checkout', 'authorizenet_vars', $authorizenet_vars );
				return;
			}
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			if ( function_exists( 'wptravel_test_mode' ) ) {
				$authorizenet_vars['testMode'] = wptravel_test_mode();
			} else {
				$authorizenet_vars['testMode'] = wp_travel_test_mode();
			}
			if ( ! empty( $settings['wp_travel_authorizenet_merchant_account_currencies'] ) ) {
				$authorizenet_vars['currencies'] = $settings['wp_travel_authorizenet_merchant_account_currencies'];
			}
			$authorizenet_vars['communicatorUrl'] = plugin_dir_url( __FILE__ ) . 'templates/communicator.html';
			$authorizenet_vars['iframeUrl']       = plugin_dir_url( __FILE__ ) . 'templates/iframe.html';

			wp_localize_script( 'wp-travel-authorize-net-checkout-views', 'wp_travel_authorizenet_vars', $authorizenet_vars );
		}
		/**
		 * Localizes a 'wp-travel-authorizenet-script' script with data for a JavaScript variable.
		 */
		public static function localize_script_forntend() {
			$authorizenet_vars = array();
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			if ( function_exists( 'wptravel_test_mode' ) ) {
				$authorizenet_vars['testMode'] = wptravel_test_mode();
			} else {
				$authorizenet_vars['testMode'] = wp_travel_test_mode();
			}
			if ( ! empty( $settings['wp_travel_authorizenet_merchant_account_currencies'] ) ) {
				$authorizenet_vars['currencies'] = $settings['wp_travel_authorizenet_merchant_account_currencies'];
			}
			$authorizenet_vars['communicatorUrl'] = plugin_dir_url( __FILE__ ) . 'templates/communicator.html';
			$authorizenet_vars['iframeUrl']       = plugin_dir_url( __FILE__ ) . 'templates/iframe.html';

			wp_localize_script( 'wp-travel-authorize-net-checkout-frontend', 'wp_travel_authorizenet_vars', $authorizenet_vars );
		}
	}

endif;
