<?php
/**
 * @package wp-travel-downloads
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add "Downloads" cpt as "WP-travel" sub-menu
 */
function wp_travel_add_downloads_under_wp_travel() {
	add_submenu_page( 'edit.php?post_type=itinerary-booking', 'Downloads', 'Downloads', 'manage_options', 'edit.php?post_type=wp_travel_downloads' );
}

if ( ! class_exists( 'WP_Travel_Downloads_Core' ) ) :
	/**
	 * Mian class for download core
	 */
	class WP_Travel_Downloads_Core {
		protected static $ABSPATH;

		protected static $PLUGIN_FILE;

		protected static $VERSION;

			/**
			 * The single instance of the class.
			 *
			 * @var WP Travel Field Editor Core
			 * @since 1.0.0
			 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Downloads_Core Instance.
		 * Ensures only one instance of WP_Travel_Downloads_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Downloads_Core()
		 * @return WP_Travel_Downloads_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 */
		public static function init( $plugin ) {
			self::$ABSPATH     = dirname( __FILE__ ) . '/';
			self::$PLUGIN_FILE = __FILE__;
			self::$VERSION     = $plugin['version'];
			define( 'WP_TRAVEL_ITINERARY_PATH', WP_CONTENT_DIR . '/' . 'wt-itinerary' . '/' );
			define( 'WP_TRAVEL_ITINERARY_TMP_PATH', WP_CONTENT_DIR . '/' . 'tmp' );

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_downloads = isset( $settings['show_wp_travel_downloads'] ) ? $settings['show_wp_travel_downloads'] : 'yes';

			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Downloads_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Downloads_Core', 'plugin_action' ) );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Downloads_Core', 'wp_travel_downloads_enqueue_styles_admin' ) );
			// if ( 'yes' !== $enable_downloads ) {
			// return;
			// }
			add_filter( 'wp_travel_display_critical_admin_notices', array( 'WP_Travel_Downloads_Core', 'display_notices' ) );
			add_action( 'wp_travel_critical_admin_notice', array( 'WP_Travel_Downloads_Core', 'notices' ) );

			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Downloads_Core', 'wp_travel_downloads_enqueue_styles_frontend' ) );
			self::wp_travel_download_includes();

			add_action( 'init', array( __CLASS__, 'wp_travel_itinerary_downloads_callback' ), 20 );
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['enable_itinerary_downloads']  = 'no';
			$settings['downloads_relative_path']     = 'yes';
			$settings['show_wp_travel_downloads']    = 'yes';
			$settings['itinerary_downloads']         = 'yes';
			$settings['show_trip_date']              = 'yes';
			$settings['show_trip_includes_excludes'] = 'yes';
			$settings['show_trip_price']             = 'yes';
			$settings['show_trip_overview']          = 'yes';
			$settings['show_trip_outline']           = 'yes';
			$settings['show_trip_itineraries']       = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_downloads';
			$field_label = __( 'WP Travel Downloads', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		/**
		 * Enqueue style and scripts for admin panel.
		 */
		public static function wp_travel_downloads_enqueue_styles_admin() {
			global $current_screen;
			$screen            = get_current_screen();
			$current_post_type = ! empty( $current_screen->post_type ) ? $current_screen->post_type : '';

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_downloads = isset( $settings['show_wp_travel_downloads'] ) ? $settings['show_wp_travel_downloads'] : 'yes';

			// if ( 'yes' === $enable_downloads ) {

				// Styles.
				wp_enqueue_style( 'main_css', plugin_dir_url( __FILE__ ) . 'assets/style.css', array(), '1.0.0' );
				wp_enqueue_style( 'select2-style' );
				wp_enqueue_style( 'font-awesome-css' );
				wp_enqueue_media();

				/**
				 * Support for email-downloadable-attachment feature.
				 *
				 * @since 1.0.8
				 */
				wp_enqueue_script( 'wp-element' );
				wp_enqueue_script( 'wp-data' );

				// Scripts.
				wp_enqueue_script( 'select2-js' );
				wp_enqueue_script( 'main_js', plugin_dir_url( __FILE__ ) . 'assets/script.js', array( 'jquery', 'jquery-ui-sortable' ), '1.0.0', true );

				$allowed_screen = array( WP_TRAVEL_POST_TYPE, 'edit-' . WP_TRAVEL_POST_TYPE, 'itinerary-enquiries' );
			if ( in_array( $screen->id, $allowed_screen ) ) {
				$deps                   = include_once sprintf( '%sapp/build/wp-travel-downloads-admin-trip-options.asset.php', self::$ABSPATH );
				$deps['dependencies'][] = 'wp-travel-admin-script';
				$deps['dependencies'][] = 'wp-travel-admin-trip-options';

				wp_enqueue_script( 'wp-travel-downloads-admin-trip-options', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-downloads-admin-trip-options.js', $deps['dependencies'], $deps['version'], true );

				// wp_enqueue_style( 'wp-travel-downloads-admin-trip-options-style', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-downloads-admin-trip-options.css', array( 'wp-components' ), $deps['version'] );
			}
			// }

			// only load on settings page [load always].
			if ( 'itinerary-booking_page_settings' === $screen->id ) {
				$deps = include_once sprintf( '%sapp/build/settings.asset.php', self::$ABSPATH );
				wp_enqueue_script( 'wp-travel-downloads-settings', plugin_dir_url( __FILE__ ) . '/app/build/settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		/**
		 * Enqueues our external font awesome stylesheet for frontend
		 *
		 * This is used for the icons that is displayed according to mime type at frontend Download tab.
		 */
		public static function wp_travel_downloads_enqueue_styles_frontend() {
			wp_enqueue_script( 'frontend_main_js', plugin_dir_url( __FILE__ ) . 'assets/download-frontend.js', array( 'jquery' ), '1.0.0', true );
			wp_enqueue_style( 'font-awesome-css' ); // @Since v1.0.4
			wp_register_style( 'wp_travel_downloads_frontend_css', plugin_dir_url( __FILE__ ) . 'assets/frontend-styles.css', array(), '1.0.0' );
			wp_enqueue_style( 'wp_travel_downloads_frontend_css' );
		}

		/**
		 * Function to include files after plugin loaded.
		 */
		public static function wp_travel_download_includes() {
			$ABSPATH = self::$ABSPATH;
			// include_once self::$ABSPATH . 'inc/wpt-downloads-public-download-list.php';
			include_once self::$ABSPATH . 'inc/wpt-downloads-cpt-and-uploads.php';
			include_once self::$ABSPATH . 'inc/wpt-downloads-select-downloads-metabox.php';
			include_once self::$ABSPATH . 'inc/downloads-email-tag-generator.php';
			include_once self::$ABSPATH . 'inc/email-downloadable-attachments.php';

			// New react includes.
			include_once self::$ABSPATH . 'core/actions/response_codes.php';
			include_once self::$ABSPATH . 'core/actions/register_post_types.php';
			include_once self::$ABSPATH . 'core/actions/trips.php';

			include_once self::$ABSPATH . 'core/helpers/downloads.php';
			include_once self::$ABSPATH . 'core/ajax/downloads.php';

			include_once self::$ABSPATH . 'app/inc/admin/metabox.php';
			include_once self::$ABSPATH . 'app/inc/frontend/downloads-tab.php';
			include_once self::$ABSPATH . 'vendor/autoload.php';
		}
		public static function wp_travel_itinerary_download_template( $trip_id, $template = 'default' ) {
			include_once self::$ABSPATH . 'inc/templates/pdf/default.php';
		}
		public static function wp_travel_itinerary_email_attachment_template( $trip_id, $template = 'default' ) {
			include self::$ABSPATH . 'inc/templates/pdf/email-attachment-pdf.php';
		}
		public static function wp_travel_itinerary_downloads_callback( $trip_id ) {
			if ( ! WP_Travel::verify_nonce( true ) ) {
				return;
			}

			if ( ! isset( $_REQUEST['download_itinerary'] ) ) {
				return;
			}
			if ( ! isset( $_REQUEST['trip_id'] ) ) {
				return;
			}
			self::generate_pdf( $_REQUEST['trip_id'] );
		}

		/**
		 * This will generate pdf. and then either download it or store it.
		 */
		public static function generate_pdf( $trip_id, $download_pdf = true ) {
			// $trip_id = $_REQUEST['trip_id'];
			$defaultConfig = ( new Mpdf\Config\ConfigVariables() )->getDefaults();
			$font_dirs     = $defaultConfig['fontDir'];

			$defaultFontConfig = ( new Mpdf\Config\FontVariables() )->getDefaults();
			$fontData          = $defaultFontConfig['fontdata'];

			$mpdf_args = array(
				'margin_top'    => 0,
				'margin_left'   => 0,
				'margin_bottom' => 0,
				'margin_right'  => 0,
				'tempDir'       => WP_TRAVEL_ITINERARY_TMP_PATH,
			);

			$mpdf_args['fontdata'] = array(
				'open-sans' => array(
					'R' => 'NotoSans-Regular.ttf',
					'B' => 'NotoSans-Bold.ttf',
				),
			);

			$mpdf_args['fontDir'] = array_merge(
				$font_dirs,
				array(
					__DIR__ . '/fonts',
				)
			);

			$mpdf = new \Mpdf\Mpdf($mpdf_args);
			ob_start();
			self::wp_travel_itinerary_download_template( $trip_id );
			$html = ob_get_contents();
			ob_end_clean();
			// echo $html;die;
			$mpdf->AddPage();
			/**
			 * @since 5.5
			 * fixed download using chinies lan
			 */
			
			$mpdf->useAdobeCJK      = true;
			$mpdf->autoLangToFont   = true;
			$mpdf->autoScriptToLang = true;

			$mpdf->WriteHTML( $html );
			$dir = trailingslashit( WP_TRAVEL_ITINERARY_PATH );

			$trips_name            = get_the_title( $trip_id );
			$downloadable_filename = $trips_name . '.pdf';
			if ( ! $download_pdf ) {
				$mpdf->Output( $dir . $downloadable_filename, 'F' ); // Store it in file.
			} else {
				$mpdf->Output( $trips_name . '.pdf', 'D' ); // download pdf.
			}
		}

		public static function notices() {
			// Invoice dir
			if ( ! file_exists( WP_TRAVEL_ITINERARY_PATH ) ) {
				$created = mkdir( WP_TRAVEL_ITINERARY_PATH, 0755, true );
				if ( ! $created ) {
					?>
					<strong> <?php esc_html_e( 'Error:', 'wp-travel-pro' ); ?></strong><?php esc_html_e( 'Unable to create directory wt-invoice.', 'wp-travel-pro' ); ?>
					<?php
				}
			}

			if ( ! is_readable( WP_TRAVEL_ITINERARY_PATH ) || ! is_writable( WP_TRAVEL_ITINERARY_PATH ) ) {
				?>
				<strong> <?php esc_html_e( 'Error:', 'wp-travel-pro' ); ?></strong>
					<?php
						echo esc_html( WP_TRAVEL_ITINERARY_PATH );
						esc_html_e( ' is not readable/writable.', 'wp-travel-pro' );
					?>
				<?php
			}

			// Invoice tmp dir
			if ( ! file_exists( WP_TRAVEL_ITINERARY_TMP_PATH ) ) {
				$created = mkdir( WP_TRAVEL_ITINERARY_TMP_PATH, 0755, true );
				if ( ! $created ) {
					?>
					<strong> <?php esc_html_e( 'Error:', 'wp-travel-pro' ); ?></strong><?php esc_html_e( 'Unable to create directory tmp.', 'wp-travel-pro' ); ?>
					<?php
				}
			}

			if ( ! is_readable( WP_TRAVEL_ITINERARY_TMP_PATH ) || ! is_writable( WP_TRAVEL_ITINERARY_TMP_PATH ) ) {
				?>
				<strong> <?php esc_html_e( 'Error:', 'wp-travel-pro' ); ?></strong>
					<?php
					echo esc_html( WP_TRAVEL_ITINERARY_TMP_PATH );
					esc_html_e( ' is not readable/writable.', 'wp-travel-pro' );
					?>
				<?php
			}
		}

		public static function display_notices( $display ) {
			// Invoice dir.
			if ( ! file_exists( WP_TRAVEL_ITINERARY_PATH ) ) {
				$created = mkdir( WP_TRAVEL_ITINERARY_PATH, 0755, true );
				if ( ! $created ) {
					$display = true;
				}
			}

			if ( ! is_readable( WP_TRAVEL_ITINERARY_PATH ) || ! is_writable( WP_TRAVEL_ITINERARY_PATH ) ) {
				$display = true;
			}

			// Invoice tmp dir.
			if ( ! file_exists( WP_TRAVEL_ITINERARY_TMP_PATH ) ) {
				$created = mkdir( WP_TRAVEL_ITINERARY_TMP_PATH, 0755, true );
				if ( ! $created ) {
					$display = true;
				}
			}

			if ( ! is_readable( WP_TRAVEL_ITINERARY_TMP_PATH ) || ! is_writable( WP_TRAVEL_ITINERARY_TMP_PATH ) ) {
				$display = true;
			}
			return $display;
		}
		/**
		 * This will generate email attachment pdf for sending itinerary pdf.
		 * @since 6.5.1
		 */
		public static function email_attachment_generate_pdf( $trip_id, $download_pdf = false ) {
			// $trip_id = $_REQUEST['trip_id'];
			$defaultConfig = ( new Mpdf\Config\ConfigVariables() )->getDefaults();
			$font_dirs     = $defaultConfig['fontDir'];

			$defaultFontConfig = ( new Mpdf\Config\FontVariables() )->getDefaults();
			$fontData          = $defaultFontConfig['fontdata'];

			$mpdf = new \Mpdf\Mpdf(
				array(
					'margin_top'    => 0,
					'margin_left'   => 0,
					'margin_bottom' => 0,
					'margin_right'  => 0,
					'tempDir'       => WP_TRAVEL_ITINERARY_TMP_PATH,
					'fontdata'      => array(
						'open-sans' => array(
							'R' => 'NotoSans-Regular.ttf',
							'B' => 'NotoSans-Bold.ttf',
							// 'I'  => 'OpenSans-Italic.ttf',
							// 'BI' => 'OpenSans-BoldItalic.ttf',
						),
					),
					'fontDir'       => array_merge(
						$font_dirs,
						array(
							__DIR__ . '/fonts',
						)
					),
				)
			);
			ob_start();
			self::wp_travel_itinerary_email_attachment_template( $trip_id );
			$html = ob_get_contents();
			ob_end_clean();

			$mpdf->AddPage();
			/**
			 * @since 5.5
			 * fixed download using chinies lan
			 */

			$mpdf->useAdobeCJK      = true;
			$mpdf->autoLangToFont   = true;
			$mpdf->autoScriptToLang = true;
			
			$mpdf->WriteHTML( $html );
			$dir = trailingslashit( WP_TRAVEL_ITINERARY_PATH );

			$trips_name            = get_the_title( $trip_id );
			$downloadable_filename = $trips_name . '.pdf';
			if ( ! $download_pdf ) {
				$mpdf->Output( $dir . $downloadable_filename, 'F' ); // Store it in file.
			} else {
				$mpdf->Output( $trips_name . '.pdf', 'D' ); // download pdf.
			}
		}
	}

endif;
