<?php
/**
 * Modules core file.
 *
 * @package wp-travel-here-map-core
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Custom Filters Core.
if ( ! class_exists( 'WP_Travel_Here_Map_Core' ) ) :
	/**
	 * WP Travel Here Map Core.
	 */
	class WP_Travel_Here_Map_Core {

		/**
		 * Absolute path to core
		 *
		 * @var string
		 */
		protected static $abspath;

		/**
		 * Undocumented variable
		 *
		 * @var [type]
		 */
		protected static $plugin_file;

		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Here Map Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Here_Map_Core Instance.
		 * Ensures only one instance of WP_Travel_Here_Map_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Here_Map_Core()
		 * @return WP_Travel_Here_Map_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param Array $params Core class init paramerters.
		 */
		public static function init( $params ) {
			self::$abspath     = dirname( __FILE__ );
			self::$plugin_file = __FILE__;
			self::$version     = $params['version'];

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_here_map = isset( $settings['show_wp_travel_here_map'] ) ? $settings['show_wp_travel_here_map'] : 'yes';

			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Here_Map_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Here_Map_Core', 'plugin_action' ) );
			if ( is_admin() ) : // load heremap settings.
				add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_settings_scripts' ) );
			endif;
			// if ( 'yes' !== $enable_here_map ) {
			// return;
			// }
			self::includes();
			add_action( 'init', array( __CLASS__, 'init_hooks' ) );
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_here_map'] = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_here_map';
			$field_label = __( 'WP Travel Here Map', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		/**
		 * Includes required files.
		 */
		public static function includes() {
			// Settings Helpers
			include sprintf( '%s/core/helpers/settings.php', self::$abspath );
		}

		/**
		 * Init Hooks.
		 */
		public static function init_hooks() {
			add_action( 'wp_travel_settings_after_currency', array( __CLASS__, 'add_settings' ), 15, 2 );
			add_action( 'wp_travel_before_save_settings', array( __CLASS__, 'push_settings' ), 15, 2 );
			add_filter( 'wp_travel_maps', array( __CLASS__, 'add_option' ) );

			$get_maps = function_exists( 'wptravel_get_maps' ) ? wptravel_get_maps() : wp_travel_get_maps();

			if ( $get_maps['selected'] !== 'wp-travel-here-map' ) {
				return;
			}

			if ( is_admin() ) :
				add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_scripts' ) );
				add_action( 'wp_travel_admin_map_area', array( __CLASS__, 'wp_travel_location_tab' ), 20, 2 );
				add_filter( 'wp_travel_localize_gallery_data', array( __CLASS__, 'localize_map_data' ) );
				add_filter( 'upload_mimes', array( __CLASS__, 'svg_support' ) );
				add_filter( 'wp_travel_save_trip_metas', array( __CLASS__, 'add_meta_key_values' ) ); // Updates trip meta array.
			endif;

			// Loader Hooks.
			add_action( 'wp_enqueue_scripts', array( __CLASS__, 'enqueue_scripts_views' ) );
			// add_action( 'wp_travel_single_trip_after_header', array( __CLASS__, 'wp_travel_frontend_map' ), 21 );
			add_action( 'wptravel_trip_map_wp-travel-here-map', array( __CLASS__, 'wp_travel_frontend_map' ), 21 );

			add_filter( 'wptravel_frontend_data', array( __CLASS__, 'localize_map_data' ) );
		}

		/**
		 * Adds Option to the WP Travel Map Selector
		 *
		 * @param array $wp_travel_maps Map Option array.
		 * @return array $wp_travel_maps Array with added option.
		 */
		public static function add_option( $wp_travel_maps ) {
			$wp_travel_maps['wp-travel-here-map'] = __( 'Here Map', 'wp-travel-pro' );
			return $wp_travel_maps;
		}

		/**
		 * Add Settings For WP Travel Here Map to WP Travel Setting Page.
		 *
		 * @param string $tab Tab key.
		 * @param mixed  $args Arguments.
		 */
		public static function add_settings( $tab, $args ) {
			if ( 'general' !== $tab ) {
				return;
			}
			include plugin_dir_path( __FILE__ ) . 'views/wp-travel-here-map-settings-tab.php';
		}

		/**
		 * Appends additonal WP Travel Here Map settings to the Default WP Travel Settings.
		 *
		 * @param array $settings WP Travel Settings array.
		 */
		public static function push_settings( $settings ) {
			$settings['wpt_here_map_enable_app_key']            = ! empty( $_POST['wpt_here_map_enable_app_key'] ) ? $_POST['wpt_here_map_enable_app_key'] : '';
			$settings['wpt_here_map_app_key']               	= ! empty( $_POST['wpt_here_map_app_key'] ) ? $_POST['wpt_here_map_app_key'] : '';
			$settings['wpt_here_map_app_id']               		= ! empty( $_POST['wpt_here_map_app_id'] ) ? $_POST['wpt_here_map_app_id'] : '';
			$settings['wpt_here_map_map_icon']             		= ! empty( $_POST['wpt_here_map_map_icon'] ) ? $_POST['wpt_here_map_map_icon'] : '';
			$settings['markerselector']                    		= ! empty( $_POST['markerselector'] ) ? $_POST['markerselector'] : '';
			$settings['wpt_here_map_map_icon_size_width']  		= ! empty( $_POST['wpt_here_map_map_icon_size_width'] ) ? $_POST['wpt_here_map_map_icon_size_width'] : '';
			$settings['wpt_here_map_map_icon_size_height'] 		= ! empty( $_POST['wpt_here_map_map_icon_size_height'] ) ? $_POST['wpt_here_map_map_icon_size_height'] : '';
			$settings['wpt_here_map_app_code']             		= ! empty( $_POST['wpt_here_map_app_code'] ) ? $_POST['wpt_here_map_app_code'] : '';
			$settings['wpt_here_map_zoom_level']           		= ! empty( $_POST['wpt_here_map_zoom_level'] ) ? $_POST['wpt_here_map_zoom_level'] : '';

			return $settings;
		}

		public static function enqueue_scripts() {
			$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : 'min.';

			$protocol = 'https';

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			// since v6.5.1 - backward compability 
			if( isset($settings['wpt_here_map_enable_app_key']) && $settings['wpt_here_map_enable_app_key'] == 'yes' ){
				$api_version = '3.1';
			}else{
				$api_version = '3.0';
			}

			wp_enqueue_style( 'here-map-ui', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-ui.css", array(), $api_version, 'all' );

			/**
			 * This module contains the core functionality for rendering maps, map layers and map objects as
			 * well as utilities used in other modules. This module forms the core of the API and is a
			 * dependency of all other modules.
			 */
			wp_register_script( 'here-maps-core', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-core.js", array(), $api_version, true );

			/**
			 * This module supports map tile retrieval, routing, geocoding, etc.
			 */
			wp_register_script( 'here-maps-service', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-service.js", array( 'here-maps-core' ), $api_version, true );

			/**
			 * This module provides a set of pre-built cross-browser UI components (such as base map settings,
			 * zoom control, map scale) that can be added to the map. It also provides functionality to display
			 * location-specific info bubbles with arbitrary HTML content on the map.
			 */
			wp_register_script( 'here-maps-ui', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-ui.js", array( 'here-maps-core' ), $api_version, true );

			/**
			 * This module provides functionality to enable user-interaction with the map, including map
			 * panning and pinch-to-zoom. The module normalizes various browser event systems (mouse events,
			 * touch events, pointer events) to offer unified events that can be used across desktop and mobile
			 * environments.
			 */
			wp_register_script( 'here-maps-mapevents', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-mapevents.js", array( 'here-maps-core' ), $api_version, true );

			wp_register_script( 'here-maps-places', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-places.js", array( 'here-maps-core' ), $api_version, true );

			$switch_to_react = true;
			if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
				$switch_to_react = wptravel_is_react_version_enabled(); // Just to include backward compatibility for now.
			}
			if ( $switch_to_react ) {
				$screen         = get_current_screen();
				$allowed_screen = array( WP_TRAVEL_POST_TYPE, 'edit-' . WP_TRAVEL_POST_TYPE, 'itinerary-enquiries' );
				if ( in_array( $screen->id, $allowed_screen ) ) {
					$deps = include_once sprintf( '%sapp/build/wp-travel-here-map-admin-trip-options.asset.php', plugin_dir_path( __FILE__ ) );

					$deps['dependencies'] = array_merge( $deps['dependencies'], array( 'here-maps-core', 'here-maps-service', 'here-maps-ui', 'here-maps-mapevents' ) );
					wp_enqueue_script( 'wp-travel-here-map-admin-trip-options', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-here-map-admin-trip-options.js', $deps['dependencies'], $deps['version'], true );
					wp_enqueue_style( 'wp-travel-here-map-admin-trip-options-style', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-here-map-admin-trip-options.css', array( 'wp-components' ), $deps['version'] );
				}
			} else {
				wp_enqueue_style( 'wp-travel-here-map', plugin_dir_url( __FILE__ ) . 'assets/css/wp-travel-here-map-admin.css', array( 'here-map-ui' ), '1.0.0', 'all' );
				wp_register_script( 'wp-travel-here-map', plugin_dir_url( __FILE__ ) . "assets/js/wp-travel-here-map-admin.{$min}js", array( 'jquery', 'here-maps-core', 'here-maps-service', 'here-maps-ui', 'here-maps-mapevents' ), '1.0.0', true );
			}

		}

		public static function enqueue_settings_scripts() {
			/**
			 * @since 2.0.0
			 */
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$switch_to_react = true; // force enable.
			if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
				$switch_to_react = wptravel_is_react_version_enabled();  // force enable but has filter to include backward compatibility for now.
			}
			if ( $switch_to_react ) {
				$screen = get_current_screen();
				// settings_screen.
				if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
					$is_settings_page = WP_Travel::is_page( 'settings', true );
				} else {
					$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
				}
				if ( $is_settings_page ) {
					$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
					$deps['dependencies'][] = 'jquery';
					wp_register_script( 'wp-travel-heremap-admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
					wp_localize_script( 'wp-travel-heremap-admin-settings', 'wp_travel_here_map', array( 'plugin_url' => plugin_dir_url( dirname( __FILE__ ) ) . 'here-map-core' ) ); // need to remove this latter.
					wp_enqueue_script( 'wp-travel-heremap-admin-settings' );
				}
			}

		}


		public static function enqueue_scripts_views() {
			$min      = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : 'min.';
			$protocol = 'https';
			
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			if( isset( $settings['wpt_here_map_enable_app_key'] ) && $settings['wpt_here_map_enable_app_key'] == 'yes' ){
				$api_version = '3.1';
			}else{
				$api_version = '3.0';
			}
			/**
			 * This function is provided for demonstration purposes only.
			 *
			 * An instance of this class should be passed to the run() function
			 * defined in WP_Travel_Here_Map_Loader as all of the hooks are defined
			 * in that particular class.
			 *
			 * The WP_Travel_Here_Map_Loader will then create the relationship
			 * between the defined hooks and the functions defined in this
			 * class.
			 */
			// wp_enqueue_style( 'wp-travel-here-map-views', plugin_dir_url( __FILE__ ) . 'assets/css/wp-travel-here-map-public.css', array( 'here-maps-ui-css' ), '1.0.0', 'all' );

			/**
			 * Here Map UI
			 */
			wp_enqueue_style( 'here-maps-ui-css', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-ui.css", array(), $api_version, 'all' );

			wp_register_script( 'wp-travel-here-map-views', plugin_dir_url( __FILE__ ) . "assets/js/wp-travel-here-map-public.{$min}js", array( 'jquery', 'here-maps-core', 'here-maps-service', 'here-maps-ui', 'here-maps-mapevents' ), '1.0.0', false );

			/**
			 * This module contains the core functionality for rendering maps, map layers and map objects as
			 * well as utilities used in other modules. This module forms the core of the API and is a
			 * dependency of all other modules.
			 */
			wp_register_script( 'here-maps-core', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-core.js", array(), $api_version, true );

			/**
			 * This module supports map tile retrieval, routing, geocoding, etc.
			 */
			wp_register_script( 'here-maps-service', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-service.js", array( 'here-maps-core' ), $api_version, true );

			/**
			 * This module provides a set of pre-built cross-browser UI components (such as base map settings,
			 * zoom control, map scale) that can be added to the map. It also provides functionality to display
			 * location-specific info bubbles with arbitrary HTML content on the map.
			 */
			wp_register_script( 'here-maps-ui', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-ui.js", array( 'here-maps-core' ), $api_version, true );

			/**
			 * This module provides functionality to enable user-interaction with the map, including map
			 * panning and pinch-to-zoom. The module normalizes various browser event systems (mouse events,
			 * touch events, pointer events) to offer unified events that can be used across desktop and mobile
			 * environments.
			 */
			wp_register_script( 'here-maps-mapevents', "{$protocol}://js.api.here.com/v3/{$api_version}/mapsjs-mapevents.js", array( 'here-maps-core' ), $api_version, true );

		}

		/**
		 * Updates $trip_meta array.
		 *
		 * @param array $trip_meta Trip Meta.
		 * @return array
		 */
		public static function add_meta_key_values( $trip_meta ) {

			$lat = isset( $_POST['wp_travel_lat'] ) ? $_POST['wp_travel_lat'] : '';
			$lng = isset( $_POST['wp_travel_lng'] ) ? $_POST['wp_travel_lng'] : '';

			$trip_meta['wp_travel_lat'] = $lat;
			$trip_meta['wp_travel_lng'] = $lng;
			return $trip_meta;
		}

		/**
		 * Callback for Backend Hook.
		 *
		 * @param string $settings WP Travel Settings.
		 * @param string $map_data WP_Travel Map Data.
		 */
		public static function wp_travel_location_tab( $settings, $map_data ) {
			$get_maps    = function_exists( 'wptravel_get_maps' ) ? wptravel_get_maps() : wp_travel_get_maps();
			$current_map = $get_maps['selected'];

			$switch_to_react = true; // force enable.
			if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
				$switch_to_react = wptravel_is_react_version_enabled();  // force enable but has filter to include backward compatibility for now.
			}

			if ( 'wp-travel-here-map' === $current_map && ! $switch_to_react ) {
				wp_enqueue_script( 'wp-travel-here-map' );
				include plugin_dir_path( __FILE__ ) . 'views/wp-travel-here-map-admin-location-tab.php';
			}
		}

		/**
		 * Localize Map Date for here map.
		 *
		 * @param array $wp_travel_gallery_data Default data array.
		 * @return array $wpt_here_map Array contianing here map data.
		 */
		public static function localize_map_data( $local ) {

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			if ( function_exists( 'wptravel_get_map_data' ) ) {
				$map_data = wptravel_get_map_data();
			} else {
				$map_data = wp_travel_get_map_data();
			}
			$zoom_level               = isset( $settings['wpt_here_map_zoom_level'] ) ? $settings['wpt_here_map_zoom_level'] : '';
			$enable_app_key                   = isset( $settings['wpt_here_map_enable_app_key'] ) ? $settings['wpt_here_map_enable_app_key'] : '';
			$app_key                 = isset( $settings['wpt_here_map_app_key'] ) ? $settings['wpt_here_map_app_key'] : '';
			$app_id                   = isset( $settings['wpt_here_map_app_id'] ) ? $settings['wpt_here_map_app_id'] : '';
			$app_code                 = isset( $settings['wpt_here_map_app_code'] ) ? $settings['wpt_here_map_app_code'] : '';
			$lat                      = ! empty( $map_data['lat'] ) ? $map_data['lat'] : '0';
			$lng                      = ! empty( $map_data['lng'] ) ? $map_data['lng'] : '0';
			$wpt_here_map_map_icon_id = isset( $settings['wpt_here_map_map_icon'] ) ? $settings['wpt_here_map_map_icon'] : '';
			$wpt_here_map_map_icon    = ( '' !== $wpt_here_map_map_icon_id ) ? wp_get_attachment_url( $wpt_here_map_map_icon_id ) : '';

			$wpt_here_map = array(
				'latlng'    => array(
					'lat' => $lat,
					'lng' => $lng,
				),
				'enable_app_key' => $enable_app_key,
				'app'       => array(
					'key'   => $app_key,
					'id'   => $app_id,
					'code' => $app_code,
				),
				'zoomLevel' => $zoom_level,
			);
			if ( is_admin() ) :
				$local['here_map'] = $wpt_here_map;
				return $local;
			endif;

			$marker_style                      = isset( $settings['markerselector'] ) ? ( $settings['markerselector'] ) : '';
			$wpt_here_map_map_icon_size_width  = isset( $settings['wpt_here_map_map_icon_size_width'] ) ? $settings['wpt_here_map_map_icon_size_width'] : '';
			$wpt_here_map_map_icon_size_height = isset( $settings['wpt_here_map_map_icon_size_height'] ) ? $settings['wpt_here_map_map_icon_size_height'] : '';

			$wpt_here_map['imageIcon']    = $wpt_here_map_map_icon;
			$wpt_here_map['markerStyle']  = $marker_style;
			$wpt_here_map['markerWidth']  = $wpt_here_map_map_icon_size_width;
			$wpt_here_map['markerHeight'] = $wpt_here_map_map_icon_size_height;
			$local['here_map']            = $wpt_here_map;
			return $local;
		}

		/**
		 * Frontend Map Content
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public static function wp_travel_frontend_map( $trip_id = 0, $data = array() ) {

			$data = function_exists( 'wptravel_get_maps' ) ? wptravel_get_maps() : wp_travel_get_maps();

			$current_map = $data['selected'];
			if ( 'wp-travel-here-map' !== $current_map ) {
				return;
			}
			if ( function_exists( 'wptravel_get_map_data' ) ) {
				$map_data = wptravel_get_map_data();
			} else {
				$map_data = wp_travel_get_map_data();
			}

			$lat = ! empty( $map_data['lat'] ) ? $map_data['lat'] : false;
			$lng = ! empty( $map_data['lng'] ) ? $map_data['lng'] : false;

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			wp_enqueue_script( 'wp-travel-here-map-views' );
			$id     = uniqid();
			$map_id = sprintf( 'wptHereMapPublic-%s', $id );
			?>
			<div class="wp-travel-map">
				<div id="<?php echo esc_attr( $map_id ); ?>" style="width:100%;height:300px"></div>
			</div>
			<script>
				jQuery(document).ready(function($) {
					// var options = {
					// 	lat : '27.693171845837',
					// 	lng : '85.281285846253',
					// }
					$( '#<?php echo esc_attr( $map_id ); ?>' ).wptravelHereMap();
				});
			</script>
			<?php
		}


		/**
		 *
		 * Allows user to upload svg file using media uploader.
		 *
		 * @param array $mimes Mime type array.
		 * @return array Array with additional mime type.
		 * @since 1.0.1
		 */
		public static function svg_support( $mimes ) {
			$mimes['svg'] = 'image/svg+xml';
			return $mimes;
		}


	}
endif;
