<?php
/**
 * Modules core file.
 *
 * @package wp-travel-import-export-core
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Custom Filters Core.
if ( ! class_exists( 'WP_Travel_Import_Export_Core' ) ) :
	/**
	 * WP Travel Custom Filters Core.
	 */
	class WP_Travel_Import_Export_Core {

		protected static $abspath;

		protected static $plugin_file;

		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP_Travel_Import_Export_Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Import_Export Instance.
		 * Ensures only one instance of WP_Travel_Import_Export is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Import_Export()
		 * @return WP_Travel_Import_Export - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $params Core class init paramerters.
		 */
		public static function init( array $params ) {
			self::$abspath     = dirname( __FILE__ );
			self::$plugin_file = __FILE__;
			self::$version     = $params['version'];

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_import_export = isset( $settings['show_wp_travel_import_export'] ) ? $settings['show_wp_travel_import_export'] : 'yes';

			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Import_Export_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Import_Export_Core', 'plugin_action' ) );
			// if ( 'yes' !== $enable_import_export ) {
			// return;
			// }

			self::includes();
			add_action( 'init', array( __CLASS__, 'init_hooks' ) );
			add_action( 'manage_posts_extra_tablenav', array( __CLASS__, 'download_booking_tablenav' ) ); // button to download.
			add_action( 'admin_init', array( __CLASS__, 'download_booking' ) ); // download csv.
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_import_export'] = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_import_export';
			$field_label = __( 'WP Travel Import Export', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php esc_html_e( sprintf( 'Show all your "%s" settings and enable its feature.', $field_label ), 'wp-travel', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		/**
		 * Includes Essential Files.
		 *
		 * @return void
		 */
		public static function includes() {
			if ( is_admin() ) {
				include sprintf( '%s/inc/class-wp-travel-import-export-ajax.php', self::$abspath );
				include sprintf( '%s/inc/class-wp-travel-csv-importer.php', self::$abspath );
				include sprintf( '%s/inc/class-wp-travel-csv-exporter.php', self::$abspath );
			}
		}

		public static function init_hooks() {
			add_action( 'admin_init', array( __CLASS__, 'register_importer' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'assets' ) );
		}

		public static function register_importer() {
			register_importer(
				'wp-travel-import-export',
				__( 'WP Travel Import Export', 'wp-travel-pro' ),
				__( 'Import WP Travel Plugin Data. Import Trips, Bookings, Enquiries, Coupons and Trip Extras data from a CSV file. ', 'wp-travel-pro' ),
				array( 'WP_Travel_CSV_Importer', 'init' )
			);

		}

		/**
		 * Register/Enqueue Scripts.
		 *
		 * @return void
		 */
		public static function assets() {
			wp_register_script( 'dropzone-js', plugin_dir_url( __FILE__ ) . 'assets/js/dropzone.js', array( 'jquery' ), '5.5.1', true );
			wp_register_script( 'sweetalert-js', 'https://cdnjs.cloudflare.com/ajax/libs/limonte-sweetalert2/7.28.2/sweetalert2.all.min.js', array( 'jquery' ), '7.28.2', true );
			wp_register_script( 'wp-travel-import-export-js', plugin_dir_url( __FILE__ ) . 'assets/js/wp-travel-import-export.js', array( 'jquery', 'dropzone-js', 'sweetalert-js' ), '1.0.0', true );
			self::localize_strings();
			if ( self::_is_wp_importer() ) {
				wp_enqueue_style( 'dropzone', plugin_dir_url( __FILE__ ) . 'assets/js/dropzone.css', array(), '5.5.1' );
				wp_enqueue_style( 'sweetalert', 'https://cdnjs.cloudflare.com/ajax/libs/limonte-sweetalert2/7.28.2/sweetalert2.min.css', array(), '7.28.2' );
				wp_enqueue_style( 'wp-travel-import-export', plugin_dir_url( __FILE__ ) . 'assets/css/wp-travel-import-export.css', array( 'dropzone', 'sweetalert' ), '1.0.0' );
				wp_enqueue_script( 'wp-travel-import-export-js' );
			}
		}

		/**
		 * Localize Strings.
		 *
		 * @return void
		 */
		public static function localize_strings() {
			$strings = array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'error'   => esc_html__( 'Import Failed !!', 'wp-travel-pro' ),
				'success' => esc_html__( 'CSV Import Successfull !!', 'wp-travel-pro' ),
				'limited' => esc_html__( 'CSV Import Successfull !!', 'wp-travel-pro' ),
			);
			wp_localize_script( 'wp-travel-import-export-js', 'wp_travel_import_export', $strings );
		}

		/**
		 * Checks if Import Page.
		 *
		 * @return void
		 */
		private static function _is_wp_importer() {
			return isset( $_GET['import'] ) && 'wp-travel-import-export' === $_GET['import'];
		}

		/**
		 * Display Download Booking in table nav.
		 *
		 * @param string $which Which section to display.
		 */
		public static function download_booking_tablenav( $which ) {
			if ( ! $which ) {
				return;
			}
			if ( 'top' === $which ) {
				$screen    = get_current_screen();
				$screen_id = $screen->id;
				if ( 'edit-itinerary-booking' !== $screen_id ) {
					return;
				}
				?>
				<a href="edit.php?post_type=itinerary-booking&download_booking=1" class="button button-secondary wp-travel-booking-download" >
					<?php esc_html_e( 'Download Bookings', 'wp-travel-pro' ); ?>
				</a>
				<script>
					jQuery(document).ready(function($) {

					});
				</script>
				<?php
			}
		}

		/**
		 * Download booking as csv.
		 */
		public static function download_booking() {
			if ( ! isset( $_REQUEST['download_booking'] ) ) {
				return;
			}
			global $wpdb;
			$post_type = 'itinerary-booking';
			$post_ids  = $wpdb->get_results(
				$wpdb->prepare( "SELECT ID, post_date from {$wpdb->posts} where post_type=%s and post_status in( 'publish', 'draft' ) order by post_date desc", $post_type )
			);
			$data      = "Booking ID, Trip Code, Contact Name, Contact Number, Contact Email, Country Code, Tour Date, Booking Status, Payment Mode, Payment Status, Booking Date \n";
			if ( is_array( $post_ids ) ) {
				foreach ( $post_ids as $booking ) {
					$booking_id = $booking->ID;
					$trip_id    = get_post_meta( $booking_id, 'wp_travel_post_id', true );
					$trip_code  = wptravel_get_trip_code( $trip_id );
					$full_name  = get_post_meta( $booking_id, 'wp_travel_fullname', true );
					if ( ! $full_name ) {
						$first_name = get_post_meta( $booking_id, 'wp_travel_fname_traveller', true );
						if ( ! $first_name ) {
							$first_name = get_post_meta( $booking_id, 'wp_travel_fname', true ); // Legacy version less than WP Travel 1.7.5 [ retriving value from old meta once. update post will update into new meta ].
						}
						$middle_name = get_post_meta( $booking_id, 'wp_travel_mname_traveller', true );
						if ( ! $middle_name ) {
							$middle_name = get_post_meta( $booking_id, 'wp_travel_mname', true );
						}
						$last_name = get_post_meta( $booking_id, 'wp_travel_lname_traveller', true );
						if ( ! $last_name ) {
							$last_name = get_post_meta( $booking_id, 'wp_travel_mname', true );
						}

						if ( is_array( $first_name ) ) { // Multiple Travelers [Extracting Traveller name from array].
							reset( $first_name );
							$first_key = key( $first_name );

							$full_name = '';
							if ( isset( $first_name[ $first_key ] ) && isset( $first_name[ $first_key ][0] ) ) {
								$full_name .= $first_name[ $first_key ][0];
							}
							if ( isset( $middle_name[ $first_key ] ) && isset( $middle_name[ $first_key ][0] ) ) {
								$full_name .= ' ' . $middle_name[ $first_key ][0];
							}
							if ( isset( $last_name[ $first_key ] ) && isset( $last_name[ $first_key ][0] ) ) {
								$full_name .= ' ' . $last_name[ $first_key ][0];
							}
						} else {
							$full_name  = $first_name;
							$full_name .= ' ' . $middle_name;
							$full_name .= ' ' . $last_name;
						}
						update_post_meta( $booking_id, 'wp_travel_fullname', $full_name );
					}

					// Booking Status.
					$status    = wptravel_get_booking_status();
					$label_key = get_post_meta( $booking_id, 'wp_travel_booking_status', true );
					if ( '' === $label_key ) {
						$label_key = 'pending';
						update_post_meta( $booking_id, 'wp_travel_booking_status', $label_key );
					}
					$booking_status = esc_attr( $status[ $label_key ]['text'] );

					// For payment data.
					$payment_info = wptravel_booking_data( $booking_id );
					$payment_id   = $payment_info['payment_id'];
					if ( is_array( $payment_id ) ) {
						if ( count( $payment_id ) > 0 ) {
							$payment_id = $payment_id[0];
						}
					}

					$label_key = get_post_meta( $payment_id, 'wp_travel_payment_status', true );
					if ( ! $label_key ) {
						$label_key = 'N/A';
						update_post_meta( $payment_id, 'wp_travel_payment_status', $label_key );
					}
					$status         = wptravel_get_payment_status();
					$payment_status = esc_attr( $status[ $label_key ]['text'] );

					$booking_data   = array();
					$booking_data[] = '#' . $booking_id; // Booking ID.
					$booking_data[] = esc_attr( $trip_code ); // Trip Code.
					$booking_data[] = esc_attr( $full_name ); // Full Name.
					$booking_data[] = is_array( get_post_meta( $booking_id, 'order_data', true )['wp_travel_phone_traveller'] ) ? esc_html( get_post_meta( $booking_id, 'order_data', true )['wp_travel_phone_traveller'][ array_key_first( get_post_meta( $booking_id, 'order_data', true )['wp_travel_phone_traveller'] ) ][0] ) : '';
					$booking_data[] = is_array( get_post_meta( $booking_id, 'order_data', true )['wp_travel_email_traveller'] ) ? esc_html( get_post_meta( $booking_id, 'order_data', true )['wp_travel_email_traveller'][ array_key_first( get_post_meta( $booking_id, 'order_data', true )['wp_travel_email_traveller'] ) ][0] ) : ''; 
					$booking_data[] = is_array( get_post_meta( $booking_id, 'order_data', true )['wp_travel_country_traveller'] ) ? esc_html( get_post_meta( $booking_id, 'order_data', true )['wp_travel_country_traveller'][ array_key_first( get_post_meta( $booking_id, 'order_data', true )['wp_travel_country_traveller'] ) ][0] ) : ''; 
					$booking_data[] = esc_html( get_post_meta( $booking_id, 'wp_travel_arrival_date', true ) );
					$booking_data[] = esc_attr( $booking_status ); // Booking Status.
					$booking_data[] = esc_html( $payment_info['payment_mode'] ); // Payment Mode.
					$booking_data[] = esc_html( $payment_status ); // Payment Status.
					$booking_data[] = esc_html( $booking->post_date ); // Date.

					$data .= implode( ',', $booking_data ) . "\n";
				}
			}

			header( 'Pragma: public' );
			header( 'Expires: 0' );
			header( 'Cache-Control: must-revalidate, post-check=0, pre-check=0' );
			header( 'Cache-Control: private', false );
			header( 'Content-Type: application/octet-stream' );
			header( 'Content-Disposition: attachment; filename="booking.csv";' );
			header( 'Content-Transfer-Encoding: binary' );

			echo $data; // @phpcs:ignore
			exit;
		}
	}
endif;
