<?php
/**
 * WP Travel Trip Inventory System Class.
 *
 * @package WP_Travel_Utilities
 */
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
if ( ! class_exists( 'WP_Travel_Util_Inventory' ) ) :
	/**
	 * WP_Travel_Util_Inventory Class.
	 */
	class WP_Travel_Util_Inventory {

		/**
		 * Constructor WP_Travel_Util_Inventory.
		 */
		public function __construct() {

			$this->init_hooks();

		}
		/**
		 * Init hooks
		 */
		public function init_hooks() {
			add_action( 'init', array( $this, 'inventory_init' ), 20 );
			add_filter( 'wp_travel_inventory_enable_status_column', array( $this, 'inventory_enable_status_column' ), 10, 2 );
			add_filter( 'wp_travel_inventory_sold_out_button', array( $this, 'inventory_sold_out_button' ), 10, 2 );

			add_action( 'wp_travel_action_after_trash_booking', array( $this, 'update_booking_availability' ) );
			add_action( 'wp_travel_action_after_untrash_booking', array( $this, 'update_booking_availability_untrash' ) );

		}

		public function inventory_init() {
			
			add_filter( 'wp_travel_inventory_data', array( $this, 'get_inventory_status_data' ), 10, 4 );
			add_filter( 'wp_travel_inventory_pax_limit', array( $this, 'inventory_get_pax_limit' ), 10, 3 );
			add_filter( 'inventory_enabled', array( $this, 'inventory_enabled' ), 10, 2 );

			$settings       = wptravel_get_settings();
			$data = WP_Travel::get_sanitize_request( 'request' );
		
			if( count($data) > 1 && class_exists( 'WP_Travel_Pro' ) && isset( $settings['selected_booking_option'] ) && count( $settings['selected_booking_option'] ) == 1 && $settings['selected_booking_option'][0] = 'booking-with-payment' ){
				if( !isset( $data['wp_travel_booking_option'] ) ){
					$data['wp_travel_booking_option'] = 'booking_with_payment';
				}
			}

			if ( isset( $data['wp_travel_booking_option'] ) ) {

				$switch_to_react = true; // force enable.
				if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
					$switch_to_react = wptravel_is_react_version_enabled();  // force enable but has filter to include backward compatibility for now.
				}
				if ( $switch_to_react ) {
				
					add_action( 'wp_travel_trip_inventory', array( $this, 'update_inventory' ) );
				} else {

					// Update Pax for single trip.
					add_action( 'wp_travel_update_trip_inventory_values', array( $this, 'update_inventory_pax_count' ), 10, 4 );
				}
			}

		}

		public function inventory_enabled( $enabled = false, $trip_id = '' ) {
			return $this->is_inventory_enabled( $trip_id );
		}

		/**
		 * Update booking availability on trash booking.
		 *
		 * @param Number $booking_id Current Booking Id.
		 *
		 * @since 2.0.3
		 */
		public function update_booking_availability( $booking_id ) {
			if ( ! $booking_id ) {
				return;
			}

			$switch_to_react = true; // force enable.
			if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
				$switch_to_react = wptravel_is_react_version_enabled();  // force enable but has filter to include backward compatibility for now.
			}
			$order_details = get_post_meta( $booking_id, 'order_items_data', true );
			if ( is_array( $order_details ) && count( $order_details ) > 0 ) {
				foreach ( $order_details as $order ) {
					$trip_id         = isset( $order['trip_id'] ) ? $order['trip_id'] : 0;
					$pax             = isset( $order['pax'] ) ? $order['pax'] : 1;
					$price_key       = isset( $order['price_key'] ) ? $order['price_key'] : '';
					$trip_start_date = isset( $order['trip_start_date'] ) ? $order['trip_start_date'] : '';

					if ( $switch_to_react ) {
						$pricing_id = isset( $order['pricing_id'] ) ? $order['pricing_id'] : 0;
						$trip_time  = isset( $order['trip_time'] ) ? $order['trip_time'] : 0;
						$meta_key   = WP_Travel_Helpers_Inventory::get_inventory_key( $pricing_id, $trip_start_date, $trip_time );
					} else {
						$meta_key = $this->get_booking_pax_count_meta_key( $price_key, $trip_start_date ); // Depricated.
					}

					// Existing booked pax.
					$booked_pax = get_post_meta( $trip_id, $meta_key, true );
					$booked_pax = ( $booked_pax ) && $booked_pax > 0 ? $booked_pax : 0;

					// decrease booked pax by current booked pax (deleted).
					$booked_pax -= $pax;
					if ( $booked_pax < 0 ) {
						$booked_pax = 0;
					}
					update_post_meta( $trip_id, $meta_key, $booked_pax );
				}
			}
		}

		/**
		 * Update booking availability on untrash booking.
		 *
		 * @param Number $booking_id Current Booking Id.
		 *
		 * @since 2.0.3
		 */
		public function update_booking_availability_untrash( $booking_id ) {
			if ( ! $booking_id ) {
				return;
			}
			$switch_to_react = true; // force enable.
			if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
				$switch_to_react = wptravel_is_react_version_enabled();  // force enable but has filter to include backward compatibility for now.
			}
			$order_details = get_post_meta( $booking_id, 'order_items_data', true );
			if ( is_array( $order_details ) && count( $order_details ) > 0 ) {
				foreach ( $order_details as $order ) {
					$trip_id         = isset( $order['trip_id'] ) ? $order['trip_id'] : 0;
					$pax             = isset( $order['pax'] ) ? $order['pax'] : 1;
					$price_key       = isset( $order['price_key'] ) ? $order['price_key'] : '';
					$trip_start_date = isset( $order['trip_start_date'] ) ? $order['trip_start_date'] : '';

					if ( $switch_to_react ) {
						$pricing_id = isset( $order['pricing_id'] ) ? $order['pricing_id'] : 0;
						$trip_time  = isset( $order['trip_time'] ) ? $order['trip_time'] : 0;
						$meta_key   = WP_Travel_Helpers_Inventory::get_inventory_key( $pricing_id, $trip_start_date, $trip_time );
					} else {
						$meta_key = $this->get_booking_pax_count_meta_key( $price_key, $trip_start_date ); // Depricated.
					}

					// Existing booked pax.
					$booked_pax = get_post_meta( $trip_id, $meta_key, true );
					$booked_pax = ( $booked_pax ) && $booked_pax > 0 ? $booked_pax : 0;

					// decrease booked pax by current booked pax (deleted).
					$booked_pax += $pax;

					update_post_meta( $trip_id, $meta_key, $booked_pax );
				}
			}
		}

		/**
		 * Check if the inventory is enabled.
		 *
		 * @param $trip_id
		 * @return bool true | false
		 */
		public function is_inventory_enabled( $trip_id, $price_key = false ) {

			if ( empty( $trip_id ) ) {

				$inventory_enabled = false;

			}

			$inventory         = get_post_meta( $trip_id, 'enable_trip_inventory', true );
			$inventory_enabled = 'yes' === $inventory ? true : false;
			return apply_filters( 'wp_travel_filter_inventory_enabled', $inventory_enabled );
		}

		/**
		 * Getters // Dont use this getter use get_pricing_option_pax_limit instead.
		 *
		 * @param number $trip_id Trip Id.
		 * get_pax_limit Get the pax limit.
		 */
		public function get_pax_limit( $trip_id ) {

			if ( empty( $trip_id ) ) {

				return false;

			}

			$limit_type = get_post_meta( $trip_id, 'wp_travel_inventory_pax_limit_type', true );

			// $pricings     = wptravel_get_trip_pricing_option( $trip_id );
			// $pricing_data = isset( $pricings['pricing_data'] ) ? $pricings['pricing_data'] : array();

			$group_size        = get_post_meta( $trip_id, 'wp_travel_group_size', true );
			$custom_group_size = get_post_meta( $trip_id, 'wp_travel_inventory_custom_max_pax', true );

			$pax_limit = 'use_group_size' === $limit_type ? absint( $group_size ) : $custom_group_size;
			// Set Group size as Default Pax Limit.
			if ( ! $pax_limit ) {
				$pax_limit = absint( $group_size );
			}
			return $pax_limit;

		}

		/**
		 * Return Pricing Options Pax Limit.
		 */
		public function get_pricing_option_pax_limit( $trip_id, $price_key ) {

			if ( empty( $trip_id ) ) {
				return false;
			}

			$inventory_enabled = $this->is_inventory_enabled( $trip_id );
			if ( ! $inventory_enabled ) {
				return;
			}
			if ( empty( $price_key ) ) {
				return $this->get_pax_limit( $trip_id );
			}

			$limit_type = get_post_meta( $trip_id, 'wp_travel_inventory_pax_limit_type', true );

			// Get Pricing variations.
			$pricing_variations = get_post_meta( $trip_id, 'wp_travel_pricing_options', true );

			if ( ! empty( $pricing_variations ) && is_array( $pricing_variations ) ) {

				$array_cols = array_column( $pricing_variations, 'price_key' ); // Price Key Array.

				// to preserve keys
				$array_cols = array_combine( array_keys( $pricing_variations ), array_column( $pricing_variations, 'price_key' ) );

				$pricing_option_key = array_search( $price_key, $array_cols, true );
				$pax_limit          = 0;
				if ( false !== $pricing_option_key ) {
					if ( 'custom_value' !== $limit_type ) {
						// Return Pax Limit.
						$pax_limit = isset( $pricing_variations[ $pricing_option_key ]['max_pax'] ) ? $pricing_variations[ $pricing_option_key ]['max_pax'] : 0;
					}
					if ( ! $pax_limit ) {
						$pax_limit = $this->get_pax_limit( $trip_id );
					}
					return $pax_limit;
				}

				return false;

			}

			return false;

		}

		function get_booking_pax_count_meta_key( $price_key = false, $trip_date = false ) {

			$meta_name = 'wp_travel_inventory_booking_pax_count';

			if ( $price_key ) {
				$meta_name .= sprintf( '_%s', $price_key ); // Added to work with trip date.
			}

			if ( $trip_date ) {
				$meta_name .= sprintf( '_%s', $trip_date ); // Added to work with trip date.
			}
			return $meta_name;
		}

		/**
		 * Get booked seats
		 */
		public function get_booking_pax_count( $trip_id, $price_key = false, $trip_date = null ) {

			if ( empty( $trip_id ) ) {
				return false;
			}

			$meta_key = $this->get_booking_pax_count_meta_key( $price_key, $trip_date );
			// Booking count value will be in new meta key since utilities 1.2.0.
			$new_booking_count = get_post_meta( $trip_id, $meta_key, true ); // Meta is used in update as well so need a function get get this meta name.

			return ( $new_booking_count ) ? $new_booking_count : 0;

		}

		/**
		 * Get Available Seats ( pax ) Note : Deprecated use WP_Travel_Helpers_Inventory::get_inventory( $args ) instead
		 */
		public function get_available_pax( $trip_id, $price_key = false, $trip_date = null ) {
			if ( empty( $trip_id ) ) {
				return false;
			}

			$max_pax = $this->get_pricing_option_pax_limit( $trip_id, $price_key ); // Get pricing maximum pax.

			$booked_pax = $this->get_booking_pax_count( $trip_id, $price_key, $trip_date );

			$available_pax = ! empty( $max_pax ) ? absint( $max_pax ) : 0;

			if ( $max_pax && $booked_pax ) {
				$available_pax = absint( $max_pax ) - absint( $booked_pax );
			}
			if ( $available_pax < 0 ) {
				$available_pax = 0;
			}

			return apply_filters( 'wp_travel_utils_available_pax', $available_pax, $trip_id, $price_key, $trip_date );

		}

		/**
		 * Setters
		 *
		 * Update booking pax count.
		 */
		public function update_booking_pax_count( $trip_id, $pax = 1, $price_key = false, $trip_date = false ) {

			if ( empty( $trip_id ) || ! $this->is_inventory_enabled( $trip_id ) || ! $this->get_pricing_option_pax_limit( $trip_id, $price_key ) ) {
				return false;
			}

			$current_pax = $this->get_booking_pax_count( $trip_id, $price_key, $trip_date );
			$updated_pax = absint( $current_pax + $pax );

			$meta_key = $this->get_booking_pax_count_meta_key( $price_key, $trip_date );

			update_post_meta( $trip_id, $meta_key, $updated_pax );
		}

		public function get_inventory_status_data( $inventory_data, $trip_id, $price_key, $trip_date = null ) {

			$available_pax = $this->get_available_pax( $trip_id, $price_key, $trip_date );
			$booked_pax    = $this->get_booking_pax_count( $trip_id, $price_key, $trip_date );
			$pax_limit     = $this->get_pricing_option_pax_limit( $trip_id, $price_key );
			$min_pax       = isset( $inventory_data['min_pax'] ) ? $inventory_data['min_pax'] : '';
			$max_pax       = isset( $inventory_data['max_pax'] ) ? $inventory_data['max_pax'] : '';
			$status_msg    = get_post_meta( $trip_id, 'wp_travel_inventory_status_message_format', true );
			if ( ! $status_msg ) {
				$status_msg = esc_html($available_pax) . ' / ' . $pax_limit . '<span>' . __( ' Pax Available', 'wp-travel-pro' ).'</span>' ;
			}

			$inventory_enabled = $this->is_inventory_enabled( $trip_id );

			if ( $inventory_enabled ) {
				$pricing_status_tags = array(
					'{available_pax}' => $available_pax,
					'{booked_pax}'    => $booked_pax,
					'{pax_limit}'     => $pax_limit,
				);

				$pricing_status_msg = str_replace( array_keys( $pricing_status_tags ), array_values( $pricing_status_tags ), $status_msg );
				$pricing_sold_out   = $available_pax === 0 || $available_pax < $min_pax ? true : false;

				if ( ! $max_pax ) {
					$max_pax = $this->get_available_pax( $trip_id, $price_key ); // no $trip_date for legacy version.
					// $max_attr = 'max=' . $max_pax;
				}
				$inventory_data['status_message'] = $pricing_status_msg;
				$inventory_data['sold_out']       = $pricing_sold_out;
				$inventory_data['available_pax']  = $available_pax;
				$inventory_data['booked_pax']     = $booked_pax;
				$inventory_data['pax_limit']      = $pax_limit;
				$inventory_data['max_pax']        = $max_pax;
			}

			return $inventory_data;
		}

		public function inventory_enable_status_column( $enable_status_column, $trip_id ) {
			$status_col        = get_post_meta( $trip_id, 'status_column_enable', true );
			return $status_col = ( $status_col && 'no' === $status_col ) ? false : true;
		}

		public function inventory_sold_out_button( $button_html, $trip_id ) {

			$sold_out_btn = get_post_meta( $trip_id, 'wp_travel_inventory_sold_out_message', true );

			if ( function_exists( 'wp_travel_get_inquiry_link' ) ) {

				$sold_out_action = get_post_meta( $trip_id, 'wp_travel_inventory_sold_out_action', true );

				if ( 'allow_trip_enquiry' === $sold_out_action ) {
					$sold_out_btn = wptravel_get_inquiry_link();
				}
			}
			return $sold_out_btn;
		}

		public function inventory_get_pax_limit( $limit, $trip_id, $price_key ) {
			return $this->get_pricing_option_pax_limit( $trip_id, $price_key );
		}

		function update_inventory_pax_count( $trip_id = null, $pax = '', $price_key = '', $trip_date = '' ) {

			if ( ! $trip_id && $trip_id == '' ) { // Legacy version compatibility.
				$trip_id = isset( $_POST['wp_travel_post_id'] ) ? $_POST['wp_travel_post_id'] : '';
			}

			if ( ! $pax && $pax == '' ) {
				$pax = isset( $_POST['wp_travel_pax'] ) ? $_POST['wp_travel_pax'] : 1;
			}

			if ( ! $this->is_inventory_enabled( $trip_id ) ) {
				return false;
			}

			// Need to use this function as callback directly.
			$this->update_booking_pax_count( $trip_id, $pax, $price_key, $trip_date );

		}
		/**
		 *
		 * @since 4.0
		 */
		function update_inventory( $args ) {

			$trip_id = $args['trip_id'];
			$pax     = $args['pax'];

			if ( ! $trip_id ) { // Legacy version compatibility.
				$trip_id = isset( $_POST['wp_travel_post_id'] ) ? $_POST['wp_travel_post_id'] : '';
			}

			if ( ! $pax ) {
				$pax = isset( $_POST['wp_travel_pax'] ) ? $_POST['wp_travel_pax'] : 1;
			}

			// error_log( print_r( $args, true ) );
			WP_Travel_Helpers_Inventory::update_inventory( $args );
		}
	}


	/**
	 * Initiate WP Travel Utilities addon.
	 */
	function wp_travel_inventory_management_core_init() {
		new WP_Travel_Util_Inventory();
	}
	wp_travel_inventory_management_core_init();
	// add_action( 'plugins_loaded', 'wp_travel_inventory_management_init' );
endif;
