<?php

/**
 * Class WP_Travel_MailChimp_Api
 */
class WP_Travel_MailChimp_Api {

	protected $version     = '3.0';
	protected $data_center = 'us2';
	protected $api_key     = null;
	protected $auth_type   = 'key';

	protected static $instance = null;

	/**
	 * @return null
	 */
	public static function getInstance() {
		return static::$instance;
	}

	/**
	 * @param $api_key
	 * @return WP_Travel_MailChimp_Api
	 */
	public static function constructInstance( $api_key ) {
		return static::$instance = new WP_Travel_MailChimp_Api( $api_key );
	}

	/**
	 * MailChimpService constructor.
	 *
	 * @param null $api_key
	 */
	public function __construct( $api_key = null ) {
		if ( ! empty( $api_key ) ) {
			$this->setApiKey( $api_key );
		}
	}

	/**
	 * @param $key
	 * @return $this
	 */
	public function setApiKey( $key ) {
		 $parts = str_getcsv( $key, '-' );

		if ( count( $parts ) == 2 ) {
			$this->data_center = $parts[1];
		}

		$this->api_key = $parts[0];

		return $this;
	}

	/**
	 * @param $dc
	 * @return $this
	 */
	public function setDataCenter( $dc ) {
		$this->data_center = $dc;

		return $this;
	}

	/**
	 * @param $version
	 * @return $this
	 */
	public function setVersion( $version ) {
		$this->version = $version;

		return $this;
	}

	/**
	 * @param bool $return_profile
	 * @param bool $throw_error
	 * @return array|bool|mixed|null|object
	 * @throws Exception
	 */
	public function ping( $return_profile = false, $throw_error = false ) {
		try {
			$profile = $this->get( '/' );
			return $return_profile ? $profile : true;
		} catch ( MailChimp_WP_Travel_Error $e ) {
			if ( $throw_error ) {
				throw $e;
			}
			return false;
		}
	}

	/**
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function getProfile() {
		return $this->get( '/' );
	}

	/**
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function getAuthorizedApps() {
		return $this->get( 'authorized-apps' );
	}

	/**
	 * @param $id
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function getAuthorizedAppDetails( $id ) {
		return $this->get( "authorized-apps/$id" );
	}

	/**
	 * @param $client_id
	 * @param $client_secret
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function linkAuthorizedApp( $client_id, $client_secret ) {
		return $this->post(
			'authorized-apps',
			array(
				'client_id'     => $client_id,
				'client_secret' => $client_secret,
			)
		);
	}

	/**
	 * @param $list_id
	 * @param $email
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function member( $list_id, $email ) {
		$hash = md5( strtolower( trim( $email ) ) );
		return $this->get( "lists/$list_id/members/$hash", array() );
	}

	/**
	 * @param $list_id
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function members( $list_id ) {
		return $this->get( "lists/$list_id/members" );
	}

	/**
	 * @param $list_id
	 * @param $email
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function deleteMember( $list_id, $email ) {
		$hash = md5( strtolower( trim( $email ) ) );
		return $this->delete( "lists/$list_id/members/$hash", array() );
	}

	/**
	 * @param $list_id
	 * @param $email
	 * @param bool    $subscribed
	 * @param array   $merge_fields
	 * @param array   $list_interests
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function subscribe( $list_id, $email, $subscribed = true, $merge_fields = array(), $list_interests = array() ) {
		$data = array(
			'email_type'    => 'html',
			'email_address' => $email,
			'status'        => ( $subscribed === true ? 'subscribed' : 'pending' ),
			'merge_fields'  => $merge_fields,
			'interests'     => $list_interests,
		);

		if ( empty( $data['merge_fields'] ) ) {
			unset( $data['merge_fields'] );
		}

		if ( empty( $data['interests'] ) ) {
			unset( $data['interests'] );
		}

		// mailchimp_debug('api.subscribe', "Subscribing {$email}", $data);
		return $this->post( "lists/$list_id/members", $data );
	}

	/**
	 * @param $list_id
	 * @param $email
	 * @param bool    $subscribed
	 * @param array   $merge_fields
	 * @param array   $list_interests
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 */
	public function update( $list_id, $email, $subscribed = true, $merge_fields = array(), $list_interests = array() ) {
		$hash = md5( strtolower( trim( $email ) ) );

		if ( $subscribed === true ) {
			$status = 'subscribed';
		} elseif ( $subscribed === false ) {
			$status = 'unsubscribed';
		} elseif ( $subscribed === null ) {
			$status = 'cleaned';
		} else {
			$status = $subscribed;
		}

		$data = array(
			'email_address' => $email,
			'status'        => $status,
			'merge_fields'  => $merge_fields,
			'interests'     => $list_interests,
		);

		if ( empty( $data['merge_fields'] ) ) {
			unset( $data['merge_fields'] );
		}

		if ( empty( $data['interests'] ) ) {
			unset( $data['interests'] );
		}

		mailchimp_debug( 'api.update_member', "Updating {$email}", $data );

		return $this->patch( "lists/$list_id/members/$hash", $data );
	}

	/**
	 * @param $list_id
	 * @param $email
	 * @param bool    $subscribed
	 * @param array   $merge_fields
	 * @param array   $list_interests
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function updateOrCreate( $list_id, $email, $subscribed = true, $merge_fields = array(), $list_interests = array() ) {
		$hash = md5( strtolower( trim( $email ) ) );

		if ( $subscribed === true ) {
			$status        = 'subscribed';
			$status_if_new = 'subscribed';
		} elseif ( $subscribed === false ) {
			$status        = 'unsubscribed';
			$status_if_new = 'pending';
		} elseif ( $subscribed === null ) {
			$status        = 'cleaned';
			$status_if_new = 'subscribed';
		} else {
			$status        = $subscribed;
			$status_if_new = 'pending';
		}

		$data = array(
			'email_address' => $email,
			'status'        => $status,
			'status_if_new' => $status_if_new,
			'merge_fields'  => $merge_fields,
			'interests'     => $list_interests,
		);

		if ( empty( $data['merge_fields'] ) ) {
			unset( $data['merge_fields'] );
		}

		if ( empty( $data['interests'] ) ) {
			unset( $data['interests'] );
		}

		// mailchimp_debug( 'api.update_or_create', "Update Or Create {$email}", $data );
		return $this->put( "lists/$list_id/members/$hash", $data );
	}

	/**
	 * @param bool $as_list
	 * @param int  $count
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function getLists( $as_list = false, $count = 100 ) {
		$result = $this->get( 'lists', array( 'count' => $count ) );

		if ( $as_list ) {
			$lists = array();
			if ( $result ) {
				$result = (object) $result;
				if ( isset( $result->lists ) && is_array( $result->lists ) ) {
					foreach ( $result->lists as $list ) {
						$list               = (object) $list;
						$lists[ $list->id ] = $list->name;
					}
				}
			}

			return $lists;
		}

		return $result;
	}

	/**
	 * @param $id
	 * @return bool
	 */
	public function hasList( $id ) {
		try {
			return (bool) $this->getList( $id );
		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * @param $id
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function getList( $id ) {
		return $this->get( 'lists/' . $id );
	}

	/**
	 * @param $id
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function deleteList( $id ) {
		 return $this->delete( 'lists/' . $id );
	}

	/**
	 * @return array|mixed
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function getListsWithMergeFields() {
		$lists = $this->getLists( true );
		foreach ( $lists as $id => $name ) {
			$lists[ $id ] = $this->mergeFields( $id, 100 );
		}

		return $lists;
	}

	/**
	 * @param $list_id
	 * @param int     $count
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function mergeFields( $list_id, $count = 10 ) {
		$result = $this->get( "lists/$list_id/merge-fields", array( 'count' => $count ) );

		return $result;
	}

	/**
	 * @param $list_id
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function getInterestGroups( $list_id ) {
		if ( empty( $list_id ) ) {
			return array();
		}
		$result = $this->get( "lists/$list_id/interest-categories" );

		return $result;
	}

	/**
	 * @param $list_id
	 * @param $group_id
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function getInterestGroupOptions( $list_id, $group_id ) {
		if ( empty( $list_id ) || empty( $group_id ) ) {
			return array();
		}
		$result = $this->get( "lists/$list_id/interest-categories/$group_id/interests" );

		return $result;
	}

	/**
	 * @param $store_id
	 * @param int           $page
	 * @param int           $count
	 * @param DateTime|null $since
	 * @param null          $campaign_id
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	public function orders( $store_id, $page = 1, $count = 10, \DateTime $since = null, $campaign_id = null ) {
		$result = $this->get(
			'ecommerce/stores/' . $store_id . '/orders',
			array(
				'start'  => $page,
				'count'  => $count,
				'offset' => ( $page * $count ),
				'since'  => ( $since ? $since->format( 'Y-m-d H:i:s' ) : null ),
				'cid'    => $campaign_id,
			)
		);

		return $result;
	}

	/**
	 * @param $store_id
	 * @return array|bool
	 */
	public function checkConnectedSite( $store_id ) {
		try {
			 return $this->get( "connected-sites/{$store_id}" );
		} catch ( MailChimp_WP_Travel_Error $e ) {
			return false;
		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * @param $store_id
	 * @return array|bool|mixed|null|object
	 */
	public function connectSite( $store_id ) {
		try {
			return $this->post( "connected-sites/{$store_id}/actions/verify-script-installation", array() );
		} catch ( MailChimp_WP_Travel_Error $e ) {
			return false;
		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * @param $url
	 * @param null $params
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	protected function delete( $url, $params = null ) {
		 $curl = curl_init();

		$options = $this->applyCurlOptions( 'DELETE', $url, $params );

		curl_setopt_array( $curl, $options );

		return $this->processCurlResponse( $curl );
	}

	/**
	 * @param $url
	 * @param null $params
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	protected function get( $url, $params = null ) {
		$curl = curl_init();

		$options = $this->applyCurlOptions( 'GET', $url, $params );

		curl_setopt_array( $curl, $options );

		return $this->processCurlResponse( $curl );
	}

	/**
	 * @param $url
	 * @param $body
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 */
	protected function patch( $url, $body ) {
		// process the patch request the normal way
		$curl = curl_init();

		$json = json_encode( $body );

		$options = $this->applyCurlOptions(
			'PATCH',
			$url,
			array(),
			array(
				'Expect:',
				'Content-Length: ' . strlen( $json ),
			)
		);

		$options[ CURLOPT_POSTFIELDS ] = $json;

		curl_setopt_array( $curl, $options );

		return $this->processCurlResponse( $curl );
	}

	/**
	 * @param $url
	 * @param $body
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	protected function post( $url, $body ) {
		$curl = curl_init();

		$json = json_encode( $body );

		$options = $this->applyCurlOptions(
			'POST',
			$url,
			array(),
			array(
				'Expect:',
				'Content-Length: ' . strlen( $json ),
			)
		);

		$options[ CURLOPT_POSTFIELDS ] = $json;

		curl_setopt_array( $curl, $options );

		return $this->processCurlResponse( $curl );
	}

	/**
	 * @param $url
	 * @param $body
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	protected function put( $url, $body ) {
		$curl = curl_init();

		$json = json_encode( $body );

		$options = $this->applyCurlOptions(
			'PUT',
			$url,
			array(),
			array(
				'Expect:',
				'Content-Length: ' . strlen( $json ),
			)
		);

		$options[ CURLOPT_POSTFIELDS ] = $json;

		curl_setopt_array( $curl, $options );

		return $this->processCurlResponse( $curl );
	}

	/**
	 * @param string     $extra
	 * @param null|array $params
	 * @return string
	 */
	protected function url( $extra = '', $params = null ) {
		$url = "https://{$this->data_center}.api.mailchimp.com/{$this->version}/";

		if ( ! empty( $extra ) ) {
			$url .= $extra;
		}

		if ( ! empty( $params ) ) {
			$url .= '?' . ( is_array( $params ) ? http_build_query( $params ) : $params );
		}

		return $url;
	}

	/**
	 * @param $method
	 * @param $url
	 * @param $body
	 * @return array|WP_Error
	 */
	protected function sendWithHttpClient( $method, $url, $body ) {
		return _wp_http_get_object()->request(
			$this->url( $url ),
			array(
				'method'  => strtoupper( $method ),
				'headers' => array(
					'Authorization' => 'Basic ' . base64_encode( 'mailchimp:' . $this->api_key ),
					'Content-Type'  => 'application/json',
				),
				'body'    => json_encode( $body ),
			)
		);
	}

	/**
	 * @param $method
	 * @param $url
	 * @param array  $params
	 * @param array  $headers
	 * @return array
	 */
	protected function applyCurlOptions( $method, $url, $params = array(), $headers = array() ) {
		$env = wp_travel_mailchimp_environment_variables();

		$curl_options = array(
			CURLOPT_USERPWD        => "mailchimp:{$this->api_key}",
			CURLOPT_CUSTOMREQUEST  => strtoupper( $method ),
			CURLOPT_URL            => $this->url( $url, $params ),
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING       => '',
			CURLOPT_MAXREDIRS      => 10,
			CURLOPT_TIMEOUT        => 30,
			CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
			CURLINFO_HEADER_OUT    => true,
			CURLOPT_HTTPHEADER     => array_merge(
				array(
					'content-type: application/json',
					"user-agent: MailChimp for WP Travel/{$env->version}; PHP/{$env->php_version}; WordPress/{$env->wp_version};",
				),
				$headers
			),
		);

		// if we have a dedicated IP address, and have set a configuration for it, we'll use it here.
		if ( defined( 'MAILCHIMP_USE_OUTBOUND_IP' ) ) {
			$curl_options[ CURLOPT_INTERFACE ] = MAILCHIMP_USE_OUTBOUND_IP;
		}

		return $curl_options;
	}

	/**
	 * @param $curl
	 * @return array|mixed|null|object
	 * @throws Exception
	 * @throws MailChimp_WP_Travel_Error
	 * @throws MailChimp_WP_Travel_ServerError
	 */
	protected function processCurlResponse( $curl ) {
		$response = curl_exec( $curl );

		$err  = curl_error( $curl );
		$info = curl_getinfo( $curl );
		curl_close( $curl );

		if ( $err ) {
			throw new MailChimp_WP_Travel_Error( 'CURL error :: ' . $err, '500' );
		}

		$data = json_decode( $response, true );

		if ( empty( $info ) || ( $info['http_code'] >= 200 && $info['http_code'] <= 400 ) ) {
			if ( is_array( $data ) ) {
				try {
					$this->checkForErrors( $data );
				} catch ( \Exception $e ) {
					throw $e;
				}
			}
			return $data;
		}

		if ( $info['http_code'] >= 400 && $info['http_code'] <= 500 ) {
			throw new MailChimp_WP_Travel_Error( $data['title'] . ' :: ' . $data['detail'], $data['status'] );
		}

		if ( $info['http_code'] >= 500 ) {
			throw new MailChimp_WP_Travel_ServerError( $data['detail'], $data['status'] );
		}

		return null;
	}

	/**
	 * @param array $data
	 * @return bool
	 * @throws MailChimp_WP_Travel_Error
	 */
	protected function checkForErrors( array $data ) {
		// if we have an array of error data push it into a message
		if ( isset( $data['errors'] ) ) {
			$message = '';
			foreach ( $data['errors'] as $error ) {
				$message .= '<p>' . $error['field'] . ': ' . $error['message'] . '</p>';
			}
			throw new MailChimp_WP_Travel_Error( $message, $data['status'] );
		}

		// make sure the response is correct from the data in the response array
		if ( isset( $data['status'] ) ) {
			$respon = (string) $data['status'];
			if ( strpos( $respon, '4' ) == true ) {
				$details      = isset( $data['detail'] ) ? $data['detail'] : '';
				$status_error = isset( $data['status'] ) ? $data['status'] : '';
				if ( ! empty( $details ) ) {
					throw new MailChimp_WP_Travel_Error( $data['detail'], $status_error );
				}
			}
		}
		return false;
	}
}
