<?php
/**
 * Modules core file.
 *
 * @package wp-travel-mailchimp-core
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Mailchimp Core.
if ( ! class_exists( 'WP_Travel_Mailchimp_Core' ) ) :
	/**
	 * WP Travel Mailchimp Core.
	 */
	class WP_Travel_Mailchimp_Core {

		const WP_TRAVEL_MAILCHIMP_HANDLE = 'wp_travel_mailchimp_';

		protected static $abspath;

		protected static $plugin_file;

		protected static $version;

		protected static $mc;

		protected static $mc_settings;

		protected static $mailchimp_api_key;

		protected static $mailchimp_list_id;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel API
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Mailchimp_Core Instance.
		 * Ensures only one instance of WP_Travel_Mailchimp_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Mailchimp_Core()
		 * @return WP_Travel_Mailchimp_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param Array $params Core class init paramerters.
		 */
		public static function init( $params ) {
			self::$abspath     = $params['abspath'] . 'inc/modules/mailchimp-core/';
			self::$plugin_file = __FILE__;
			self::$version     = $params['version'];

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_mailchimp = isset( $settings['show_wp_travel_mailchimp'] ) ? $settings['show_wp_travel_mailchimp'] : 'yes';

			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Mailchimp_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Mailchimp_Core', 'plugin_action' ) );
			add_action( 'init', array( __CLASS__, 'init_hooks' ) );
			// if ( 'yes' !== $enable_mailchimp ) {
			// 	return;
			// }

			self::$mc_settings = isset( $settings['mailchimp'] ) ? $settings['mailchimp'] : array();
			self::includes();
			self::init_mc();
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_mailchimp'] = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_mailchimp';
			$field_label = __( 'WP Travel Mailchimp', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php esc_html_e( sprintf( 'Show all your "%s" settings and enable its feature.', $field_label ), 'wp-travel', 'wp-travel-mailchimp', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		/**
		 * Registers/Enqueues Scripts.
		 */
		public static function assets() {
			$screen = get_current_screen();
			// @since WP Travel 4.4.2
			if (  method_exists( 'WP_Travel',  'is_page' )  ) {
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_MAILCHIMP_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
				wp_localize_script( self::WP_TRAVEL_MAILCHIMP_HANDLE . 'admin-settings', 'wp_travel_mailchimp', array( 'ajaxUrl' => admin_url( 'admin-ajax.php' ) ) );
			}
			wp_register_script( 'wp-travel-mailchimp-script', plugin_dir_url( __FILE__ ) . 'assets/js/wp-travel-mailchimp-script.js', array( 'jquery' ), '1.0.0', true );
			wp_localize_script( 'wp-travel-mailchimp-script', 'wp_travel_mailchimp', array( 'ajaxUrl' => admin_url( 'admin-ajax.php' ) ) );
		}

		/**
		 * Includes essential files.
		 *
		 * @return void
		 */
		private static function includes() {
			include_once self::$abspath . 'inc/wp-travel-mailchimp-settings.php';
			include_once self::$abspath . 'inc/functions.php';
			include_once self::$abspath . 'inc/api/errors/class-mailchimp-error.php';
			include_once self::$abspath . 'inc/api/errors/class-mailchimp-rate-limit-error.php';
			include_once self::$abspath . 'inc/api/errors/class-mailchimp-server-error.php';
			include_once self::$abspath . 'inc/api/class-wp-travel-mailchimp-api.php';
		}

		/**
		 * Initialize Mailchimp.
		 *
		 * @return void
		 */
		private static function init_mc() {
			$settings                = self::$mc_settings;
			self::$mailchimp_api_key = ! empty( $settings['api_key'] ) ? $settings['api_key'] : null;
			self::$mailchimp_list_id = ! empty( $settings['list_id'] ) ? $settings['list_id'] : null;
			if ( self::$mailchimp_api_key ) {
				self::$mc = WP_Travel_MailChimp_Api::constructInstance( self::$mailchimp_api_key );
			}
		}

		/**
		 * Initializes hooks.
		 */
		public static function init_hooks() {
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'assets' ) );
			if ( ! self::$mc instanceof WP_Travel_MailChimp_Api ) {
				return;
			}
			add_action( 'wp_travel_after_enquiries_email_sent', array( __CLASS__, 'enquiry_form_data' ), 11, 4 );
			add_action( 'wp_travel_after_frontend_booking_save', array( __CLASS__, 'checkout_form_data' ) );
		}

		/**
		 * Add email to the Mailchimp List via API call.
		 *
		 * @param string $customer_email Email to be added.
		 * @param array  $merge_fields Additional Field to be added.
		 * @return void
		 */
		private static function add_to_mailchimp_list( $customer_email, $merge_fields = array() ) {
			try {
				self::$mc->updateOrCreate( self::$mailchimp_list_id, $customer_email, self::subscribed_status(), $merge_fields );
			} catch ( MailChimp_WP_Travel_Error $e ) {
				$message = 'Message: ' . $e->getMessage();
			} catch ( MailChimp_WP_Travel_ServerError $e ) {
				$message = 'Message: ' . $e->getMessage();
			}
		}

		/**
		 * Prepare email to be added from inquiry form.
		 *
		 * @param [type] $admin_email Admin Email.
		 * @param [type] $customer_email Customer's Email.
		 * @param [type] $formdata Enquiry Form Data.
		 * @param [type] $enquiry_id Enquiry ID.
		 * @return void
		 */
		public static function enquiry_form_data( $admin_email, $customer_email, $formdata, $enquiry_id ) {
			if ( ! self::is_opt_in_enabled() && ! isset( $formdata['wp_travel_subscribe'] ) ) {
				return;
			}

			$merge_fields          = array();
			$merge_fields['FNAME'] = ! empty( $formdata['wp_travel_enquiry_name'] ) ? $formdata['wp_travel_enquiry_name'] : '';
			self::add_to_mailchimp_list( $customer_email, $merge_fields );
		}

		/**
		 * Prepare email to be added from Checkout form.
		 *
		 * @param [type] $booking_id Booking ID.
		 * @return void
		 */
		public static function checkout_form_data( $booking_id ) {
			if ( ! isset( $_POST['wp_travel_subscribe'] ) && ! self::is_opt_in_enabled() ) {
				return;
			}
			if ( isset( $_POST['wp_travel_email_traveller'] ) ) {
				foreach ( $_POST['wp_travel_email_traveller'] as $trip_emails ) {
					foreach ( $trip_emails as $email ) {
						self::add_to_mailchimp_list( $email );
					}
				}
			}
		}

		/**
		 * Checks if Opt-in enabled.
		 *
		 * @return boolean
		 */
		private static function is_opt_in_enabled() {
			$settings = self::$mc_settings;
			return isset( $settings['opt_in'] ) && '1' === $settings['opt_in'];
		}

		/**
		 * Returns Subscribed status according to the opt-in option.
		 *
		 * @return string
		 */
		private static function subscribed_status() {

			if ( self::is_opt_in_enabled() ) {
				return false;
			} else {
				return true;
			}
		}
	}
endif;
