<?php
/**
 * @package multiple-currency-core/inc/
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Multiple_Currency_API_Data' ) ) {
	/**
	 * This class will process the api data.
	 */
	class WP_Travel_Multiple_Currency_API_Data {

		/**
		 * Checks if WP Travel Currency Exchange Rate plugin is activated.
		 * Also checks for addon settings if wp travel pro is activate.
		 *
		 * @return boolean
		 */
		public static function is_active_currency_exchange_rates() {
			if ( class_exists( 'WP_Travel_Pro' ) ) {
				if ( function_exists( 'wptravel_get_settings' ) ) {
					$settings = wptravel_get_settings();
				} else {
					$settings = wp_travel_get_settings();
				}
				return isset( $settings['show_wp_travel_currency_exchange_rates'] ) && ( 'yes' === $settings['show_wp_travel_currency_exchange_rates'] ) ? true : false;
			}
		}

		/**
		 * Return the api url ( premium or standard ) or returns error message if $check_error is true.
		 *
		 * @param string  $api_key    API key [Required].
		 * @param boolean $is_premium If the api key is premium or standard. Default is false.
		 *
		 * @return mixed
		 */
		public static function get_api_url( $api_key, $is_premium = false ) {

			if ( empty( $api_key ) ) {
				return;
			}

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$use_api_layer_fixer_api = isset( $settings['use_api_layer_fixer_api'] ) && 'yes' === $settings['use_api_layer_fixer_api'];

			$api_protocal = 'http://';
			if ( $is_premium ) {
				$api_protocal = 'https://';
			}

			$api_url = $api_protocal . 'data.fixer.io/api/latest?access_key=' . $api_key . '&format=1';
			if ( $use_api_layer_fixer_api ) {
				$api_url = $api_protocal . 'api.apilayer.com/fixer/latest?apikey=' . $api_key . '&format=1';
			}
			return $api_url;
		}

		/**
		 * Returns the API data
		 *
		 * @return array $api_data
		 */
		public static function get_api_data() {

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$multiple_currency_api_key        = ! empty( $settings['exchange_rate_api_key'] ) && isset( $settings['exchange_rate_api_key'] ) ? $settings['exchange_rate_api_key'] : '';
			$multiple_currency_api_is_premium = ! empty( $settings['exchange_rate_api_key_premium_checkbox'] ) && isset( $settings['exchange_rate_api_key_premium_checkbox'] ) ? $settings['exchange_rate_api_key_premium_checkbox'] : '';

			if ( empty( $multiple_currency_api_key ) ) {
				return;
			}
			$multiple_currency_api_is_premium = ( 'yes' === $multiple_currency_api_is_premium ) ? true : false;

			$api_link      = self::get_api_url( $multiple_currency_api_key, $multiple_currency_api_is_premium );
			$api_link_data = curl_init( $api_link );
			curl_setopt( $api_link_data, CURLOPT_RETURNTRANSFER, true );
			$json = curl_exec( $api_link_data );
			curl_close( $api_link_data );
			$api_data = json_decode( $json, true );

			return $api_data;
		}

		/**
		 * Returns API error message.
		 *
		 * @param string  $api_key
		 * @param boolean $is_premium
		 * @return string
		 */
		public static function subscription_error_message() {
			$api_data = self::get_api_data();
			return isset( $api_data['error']['info'] ) ? $api_data['error']['info'] : '';
		}

		/**
		 * Returns the "Click here to get API Access key" if field is empty,
		 * or returns the "Click here to test your API key" if no error in api.
		 * or returns error text.
		 *
		 * @return string Get API key text or error message.
		 */
		public static function get_description() {
			$description = '';
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$multiple_currency_api_key        = ! empty( $settings['exchange_rate_api_key'] ) && isset( $settings['exchange_rate_api_key'] ) ? $settings['exchange_rate_api_key'] : '';
			$multiple_currency_api_is_premium = ! empty( $settings['exchange_rate_api_key_premium_checkbox'] ) && isset( $settings['exchange_rate_api_key_premium_checkbox'] ) ? $settings['exchange_rate_api_key_premium_checkbox'] : '';

			$multiple_currency_api_is_premium = ( 'yes' === $multiple_currency_api_is_premium ) ? true : false;
			$subscription_error_message       = ! empty( self::subscription_error_message() ) ? self::subscription_error_message() : '';
			$get_api_url                      = ! empty( self::get_api_url( $multiple_currency_api_key, $multiple_currency_api_is_premium ) ) ? self::get_api_url( $multiple_currency_api_key, $multiple_currency_api_is_premium ) : '';

			if ( ! empty( $multiple_currency_api_key ) ) {
				if ( ! $multiple_currency_api_is_premium ) {
					if ( ! empty( $subscription_error_message ) ) {
						$description = '<p class="description" style="color:red;">' . esc_html( $subscription_error_message ) . '</p>';
					} else {
						$description = '<p class="description"><a href="' . $get_api_url . '" target="_blank">' . esc_html__( 'Click here', 'wp-travel-pro' ) . '</a>' . esc_html__( ' to test your API key.', 'wp-travel-pro' ) . '</p>';
					}
				} else {
					if ( ! empty( $subscription_error_message ) ) {
						$description = '<p class="description" style="color:red;">' . esc_html( $subscription_error_message ) . '</p>';
					} else {
						$description = '<p class="description"><a href="' . $get_api_url . '" target="_blank">' . esc_html__( 'Click here', 'wp-travel-pro' ) . '</a>' . esc_html__( ' to test your API key.', 'wp-travel-pro' ) . '</p>';
					}
				}
			} else {
				$description = '<p class="description"><a href="https://fixer.io/product" target="_blank">' . esc_html__( 'Click here', 'wp-travel-pro' ) . '</a>' . esc_html__( ' to get your API Access key.', 'wp-travel-pro' ) . '</p>';
			}

			return $description;
		}

		/**
		 * Retrives the geolocation data on the basis of ip address.
		 * But to be honest, don't rely on this gimmic.
		 *
		 * @todo always calling this api to fetch geolocation data. need to enhance.
		 *
		 * @return void
		 */
		public static function geolocation_data() {

			// $api_link      = 'http://www.geoplugin.net/json.gp';
			$user_ip  = isset( $_SERVER['REMOTE_ADDR'] ) ? $_SERVER['REMOTE_ADDR'] : '';
			$api_link = 'http://ip-api.com/json/';
			if ( $user_ip && ! self::is_private_ip( $user_ip ) ) {
				$api_link .= $user_ip;
			}
			$api_link = add_query_arg( 'fields', 8441855, $api_link ); // fields param has generated numeric value [src : https://ip-api.com/docs/api:json ].
			// Curl.
			$api_link_data = curl_init( $api_link );

			curl_setopt( $api_link_data, CURLOPT_RETURNTRANSFER, true );
			$json = curl_exec( $api_link_data );

			curl_close( $api_link_data );
			$geolocation_data = json_decode( $json, true );
			return $geolocation_data;
		}

		public static function is_private_ip( $ip ) {
			if ( '::1' === $ip ) {
				return true;
			}
			$pri_addrs = array(
				'10.0.0.0|10.255.255.255', // single class A network.
				'172.16.0.0|172.31.255.255', // 16 contiguous class B network.
				'192.168.0.0|192.168.255.255', // 256 contiguous class C network.
				'169.254.0.0|169.254.255.255', // Link-local address also refered to as Automatic Private IP Addressing.
				'127.0.0.0|127.255.255.255', // localhost.
			);

			$long_ip = ip2long( $ip );
			if ( $long_ip != -1 ) {

				foreach ( $pri_addrs as $pri_addr ) {
					list ($start, $end) = explode( '|', $pri_addr );
					// IF IS PRIVATE.
					if ( $long_ip >= ip2long( $start ) && $long_ip <= ip2long( $end ) ) {
						return true;
					}
				}
			}
			return false;
		}
	}
}
