<?php
/**
 * This file handles the hooks and filters for overriding price value
 * according to the addon usuability of the hooks.
 *
 * Originally the functions.php was handling all the pluggables along with individual functions which was later moved in this file
 * keeping pluggables and individual functions separate.
 *
 * @package wp-travel-multiple-currency-core
 * @since WP Travel Pro 3.0.4
 */

/**
 * Exit if accessed directly.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * Changes the symbols according to the selected currency in trip.
 *
 * @return string returns the currency symbol accordingly.
 */
function wp_travel_multiple_currency_converted_trip_symbol( $symbols, $booking_id = null ) {
	if ( empty( $booking_id ) ) {
		$booking_id = get_the_ID();
	}
	if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
		$is_dashboard_page = WP_Travel::is_page( 'dashboard' );
	} else {
		$is_dashboard_page = wp_travel_is_dashboard_page();
	}
	if ( is_admin() || $is_dashboard_page ) {
		$customer_selected_currency = get_post_meta( $booking_id, 'wp_travel_multiple_currency_customer_selected_symbol', true );
		if ( ! empty( $customer_selected_currency ) ) {

			if ( function_exists( 'wptravel_get_currency_symbol' ) ) {
				return wptravel_get_currency_symbol( $customer_selected_currency );
			} else {
				return wp_travel_get_currency_symbol( $customer_selected_currency );
			}
		} else {
			if ( function_exists( 'wptravel_get_currency_symbol' ) ) {
				return wptravel_get_currency_symbol();
			} else {
				return wp_travel_get_currency_symbol();
			}
		}
	}
	$customer_selected_currency = WP_Travel_Multiple_Currency_Selector_Dropdown::get_customer_selected_currency();
	if ( function_exists( 'wptravel_get_currency_symbol' ) ) {
		return wptravel_get_currency_symbol( $customer_selected_currency );
	} else {
		return wp_travel_get_currency_symbol( $customer_selected_currency );
	}
}
add_filter( 'wp_travel_display_currency_symbol', 'wp_travel_multiple_currency_converted_trip_symbol', 12, 2 );
add_filter( 'wp_travel_tour_extras_currency_symbol', 'wp_travel_multiple_currency_converted_trip_symbol' );

/**
 * Changes the currency code for the standard paypal.
 *
 * @param  array $args
 * @return array $args
 */
function wp_travel_multiple_currency_standard_paypal_support( $args ) {
	$args['currency_code'] = WP_Travel_Multiple_Currency_Selector_Dropdown::get_customer_selected_currency();
	/**
	 * Start converting trip extras price accordingly.
	 */
	if ( is_array( $args ) && ! empty( $args ) ) {
		$args_index = 2; // Started from 2 because trip amount is already set on 1.
		foreach ( $args as $args_key => $order_data ) {
			$amount_index = 'amount_' . $args_index;
			if ( isset( $args[ $amount_index ] ) ) {
				$args[ $amount_index ] = wp_travel_multiple_currency_convert_price( $args[ $amount_index ] );
			}
			$args_index++;
		}
	}
	return $args;
}
add_filter( 'wp_travel_paypal_args', 'wp_travel_multiple_currency_standard_paypal_support' );

/**
 * Returns the trips converted prices according to the selected currency code.
 *
 * @param  int $price
 * @return int $price
 */
function wp_travel_multiple_currency_converted_trip_price( $price ) {
	if ( empty( $price ) ) {
		return $price;
	}
	// This will return if not ajax request.
	if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
		return $price;
	}
	$price = wp_travel_multiple_currency_convert_price( $price );
	return $price;
}
// add_filter( 'wp_travel_price', 'wp_travel_multiple_currency_converted_trip_price' );
// add_filter( 'wp_travel_convert_price', 'wp_travel_multiple_currency_converted_trip_price' ); // @since WP Travel Pro 3.0.4, requires wp-travel 4.0.7, see wptravel_convert_price() in wp-travel
// add_filter( 'wp_travel_group_discount_price_override', 'wp_travel_multiple_currency_converted_trip_price' );
// add_filter( 'wp_travel_trip_extras_custom_prices', 'wp_travel_multiple_currency_converted_trip_price' );
// add_filter( 'wp_travel_trip_extras_custom_sale_prices', 'wp_travel_multiple_currency_converted_trip_price' );
add_filter( 'wp_travel_multiple_currency', 'wp_travel_multiple_currency_converted_trip_price' ); // Replace with 'wp_travel_multiple_currency_price' as 'wp_travel_multiple_currency_price' function not exists. @since WP Travel v4.3.0.

/**
 * Unset cookie value on save settings to fix base currency mismatch issue.
 */
function wtmc_clear_selected_currency_cookie() {
	if ( isset( $_COOKIE['wp_travel_multiple_currency_customerSelectedCurrency'] ) ) {
		unset( $_COOKIE['wp_travel_multiple_currency_customerSelectedCurrency'] );
		setcookie( 'wp_travel_multiple_currency_customerSelectedCurrency', null, -1, '/' );
	}
}
add_action( 'wptravel_action_after_settings_saved', 'wtmc_clear_selected_currency_cookie' );

/**
 * Supports for the payment gateways.
 *
 * @param array $wp_travel
 * @return array $wp_travel
 */
function wp_travel_multiple_currency_payment_support( $wp_travel ) {
	$customer_selected_currency            = WP_Travel_Multiple_Currency_Selector_Dropdown::get_customer_selected_currency();
	$wp_travel['payment']['currency_code'] = $customer_selected_currency;

	if ( function_exists( 'wptravel_get_currency_symbol' ) ) {
		$currency_symbol = wptravel_get_currency_symbol( $customer_selected_currency );
	} else {
		$currency_symbol = wp_travel_get_currency_symbol( $customer_selected_currency );
	}
	$wp_travel['payment']['currency_symbol'] = $currency_symbol;
	return $wp_travel;
}
add_filter( 'wptravel_frontend_data', 'wp_travel_multiple_currency_payment_support' );

/**
 * Filter for Currency Symbol accordance with the currency selected.
 *
 * @since 2.1.3 (WP Travel Pro)
 */
function wp_travel_multiple_currency_symbol( $symbol, $currency_code, $currency_symbols ) {
	if ( is_admin() ) {
		return $symbol;
	}
	$customer_selected_currency = WP_Travel_Multiple_Currency_Selector_Dropdown::get_customer_selected_currency();
	$symbol                     = ! empty( $customer_selected_currency ) && in_array( $customer_selected_currency, array_keys( $currency_symbols ), true ) ? $currency_symbols[ $customer_selected_currency ] : $symbol;
	return $symbol;
}
add_filter( 'wp_travel_currency_symbol', 'wp_travel_multiple_currency_symbol', 15, 3 );

/**
 * Filter for Currency Name accordance with the currency selected.
 *
 * @since 3.1.9 (WP Travel Pro)
 */
function wp_travel_multiple_currency_name( $currency_code ) {
	if ( is_admin() ) {
		return $currency_code;
	}
	$customer_selected_currency = WP_Travel_Multiple_Currency_Selector_Dropdown::get_customer_selected_currency();
	return $customer_selected_currency;
}
add_filter( 'wp_travel_use_currency_name', 'wp_travel_multiple_currency_name' );

/**
 * Saves the customer selected currency symbol to database according to booking id after checkout.
 * It is needed to display the currency according to the booking id for the bookings details.
 *
 * @param  int $booking_id
 * @return void
 */
function wp_travel_multiple_currency_save_selected_symbol( $booking_id ) {
	if ( empty( $booking_id ) ) {
		return $booking_id;
	}
	$customer_selected_currency = WP_Travel_Multiple_Currency_Selector_Dropdown::get_customer_selected_currency();
	if ( ! empty( $customer_selected_currency ) ) {
		update_post_meta( $booking_id, 'wp_travel_multiple_currency_customer_selected_symbol', $customer_selected_currency );
	}
}
add_action( 'wp_travel_after_frontend_booking_save', 'wp_travel_multiple_currency_save_selected_symbol' );

/**
 * This will add localized data for multiple currency.
 *
 * @param array $localized Localized data.
 */
function wptravel_multiple_currency_localized_data( $localized ) {
	$localized['_wp_travel']['conversion_rate'] = wtmc_get_conversion_rate();
	$localized['wp_travel']['conversion_rate']  = wtmc_get_conversion_rate();
	return $localized;
}

add_filter( 'wptravel_localized_data', 'wptravel_multiple_currency_localized_data' );
