<?php
/**
 * This file has all the requried code for the enabling option for multiple partial payments.
 *
 * @since 1.0.6
 * @package partial-payment-core/inc/admin/
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Multiple_Partial_Payment_Option' ) ) {
	/**
	 * Enables the option for the multiple partial payment.
	 */
	class WP_Travel_Multiple_Partial_Payment_Option {

		/**
		 * Init class.
		 *
		 * @return void
		 */
		public static function init() {
			add_filter( 'wp_travel_before_minimum_partial_payout', array( __CLASS__, 'create_buttons' ) );
			add_filter( 'wp_travel_after_minimum_partial_payout', array( __CLASS__, 'extend_partial_payments' ) );
			add_filter( 'wp_travel_partial_payment_string', array( __CLASS__, 'partial_payment_string' ), 12, 2 );
			add_filter( 'wp_travel_before_minimum_partial_payout', array( __CLASS__, 'warning_message' ) );
		}

		/**
		 * Changes the Minimum Payout text.
		 *
		 * @param  string $partial_payment_string Default partial payment label.
		 * @param  array  $settings               Saved settings array.
		 * @return string $partial_payment_string New partial payment label.
		 */
		public static function partial_payment_string( $partial_payment_string, $settings ) {
			$multiple_partial_payment_payouts = ! empty( $settings['minimum_partial_payout'] ) ? $settings['minimum_partial_payout'] : array();
			$partial_payment_string           = is_array( $multiple_partial_payment_payouts ) && count( $multiple_partial_payment_payouts ) > 1 ? __( 'Partial Payout 1 (%)', 'wp-travel-pro' ) : $partial_payment_string;
			return $partial_payment_string;
		}

		/**
		 * Creates the buttons for adding and deleting the partial payment fields.
		 *
		 * @return void
		 */
		public static function create_buttons() {
			?>
			<tr class="wp-travel-partial-payment-payouts-row">
				<th><label for=""></label></th>
				<td>
					<input type="button" class="button button-primary" id="btn_add_partial_payment" value="<?php esc_attr_e( 'Add Partial Payment', 'wp-travel-pro' ); ?>" />
					<input type="button" class="button button-primary" id="btn_delete_partial_payment" value="<?php esc_attr_e( 'Delete Partial Payment', 'wp-travel-pro' ); ?>" />
				</td>
			</tr>
			<?php
		}

		/**
		 * Add new fields and for extending the partial payment fields.
		 *
		 * @param  array $settings Saved settings array.
		 * @return void
		 */
		public static function extend_partial_payments( $settings ) {
			$multiple_partial_payment_payouts = ! empty( $settings['minimum_partial_payout'] ) ? $settings['minimum_partial_payout'] : array();
			if ( is_array( $multiple_partial_payment_payouts ) && isset( $multiple_partial_payment_payouts[0] ) ) {
				unset( $multiple_partial_payment_payouts[0] );
			}
			if ( is_array( $multiple_partial_payment_payouts ) && count( $multiple_partial_payment_payouts ) > 0 ) {
				foreach ( $multiple_partial_payment_payouts as $key => $multiple_partial_payment_payout ) {
					$payout_index           = $key + 1;
					$partial_payment_string = 'Partial Payout ' . $payout_index . ' (%)';
					?>
					<tr class="wp-travel-multiple-partial-payment-payouts wp-travel-partial-payment-payouts-row">
						<th><label for="multiple_partial_payment_payouts_output_<?php echo esc_attr( $payout_index ); ?>"><?php echo esc_html( $partial_payment_string ); ?></label></th>
						<td>
							<input type="range" min="1" max="100" step="0.01" value="<?php echo esc_attr( $multiple_partial_payment_payout ); ?>" id="multiple_partial_payment_payouts_<?php echo esc_attr( $payout_index ); ?>" data-input-id="multiple_partial_payment_payouts_output_<?php echo esc_attr( $payout_index ); ?>" class="wt-slider multiple-partial-payment-payouts-slider" />
							<label><input type="number" min="1" max="100" step="0.01" value="<?php echo esc_attr( $multiple_partial_payment_payout ); ?>" name="minimum_partial_payout[]" data-slider-id="multiple_partial_payment_payouts_<?php echo esc_attr( $payout_index ); ?>" class="multiple-partial-payment-payouts-output-field" id="multiple_partial_payment_payouts_output_<?php echo esc_attr( $payout_index ); ?>" />%</label>
							<p class="description"><?php esc_html_e( 'Minimum percent of amount to pay while booking.', 'wp-travel-pro' ); ?></p>
						</td>
					</tr>
					<?php
				}
			}
		}

		/**
		 * Calculates the percentage of all partial payment payouts.
		 *
		 * @param array $settings
		 * @return int  $total_percent
		 */
		public static function get_payout_total_percent( $settings = array() ) {
			$total_payout_percent = 0;
			$payout_percentages   = isset( $settings['minimum_partial_payout'] ) && ! empty( $settings['minimum_partial_payout'] ) ? $settings['minimum_partial_payout'] : array();
			if ( ! empty( $payout_percentages ) && is_array( $payout_percentages ) ) {
				foreach ( $payout_percentages as $payout_percent ) {
					if ( ! empty( $payout_percent ) ) {
						if ( ! is_int( $payout_percent ) ) {
							$payout_percent = (int) $payout_percent;
						}
						$total_payout_percent += $payout_percent;
					}
				}
			}
			$total_percent = $total_payout_percent;
			return $total_percent;
		}

		/**
		 * Checks if the multiple partial payment is vaild for processings.
		 *
		 * @return boolean
		 */
		public static function is_payouts_valid() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$payout_percentages   = isset( $settings['minimum_partial_payout'] ) && ! empty( $settings['minimum_partial_payout'] ) ? $settings['minimum_partial_payout'] : array();
			$total_payout_percent = self::get_payout_total_percent( $settings );
			$count_payout_percent = is_array( $payout_percentages ) ? count( $payout_percentages ) : 1;
			/**
			 * Return true if no partial payouts has been added.
			 */
			if ( 1 === $count_payout_percent ) {
				return true;
			}
			return $count_payout_percent > 0 && '100' === (string) $total_payout_percent;
		}

		/**
		 * Displays the warning message if payout is not equals to 100%.
		 *
		 * @return void
		 */
		public static function warning_message() {
			$is_payouts_valid = self::is_payouts_valid();
			if ( ! $is_payouts_valid ) {
				?>
				<tr class="wp-travel-partial-payment-payouts-row">
					<th></th>
					<td>
						<p class="wp-travel-notification notification-warning notice notice-error"><?php echo esc_html( 'Error: Total payout percent is not equals to 100%.' ); ?></p>
					</td>
				</tr>
				<?php
			}
		}
	}
}

WP_Travel_Multiple_Partial_Payment_Option::init();
