<?php
/**
 * This file has all the requried code for the frontend payment process.
 *
 * @since 1.0.6
 * @package partial-payment-core/inc/
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Travel_Partial_Payment_Frontend_Process' ) ) {
	/**
	 * Enables the option for the multiple partial payment.
	 */
	class WP_Travel_Partial_Payment_Frontend_Process {

		/**
		 * Init class.
		 *
		 * @return void
		 */
		public static function init() {
			add_action( 'wp_travel_after_frontend_booking_save', array( __CLASS__, 'save_partial_payout_array_structure' ), 12, 2 );
			add_filter( 'wp_travel_partial_payment_due_amount', array( __CLASS__, 'due_amount' ) );
			add_filter( 'partial_payment_form_fields', array( __CLASS__, 'dashboard_installment_information' ), 12, 2 );
		}

		/**
		 * Saves the partial payout array structure during checkout.
		 *
		 * @return void
		 */
		public static function save_partial_payout_array_structure( $booking_id, $first_key ) {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$trip_id        = get_post_meta( $booking_id, 'wp_travel_post_id', true );
			$use_global     = get_post_meta( $trip_id, 'wp_travel_minimum_partial_payout_use_global', true );
			$custom_payouts = get_post_meta( $trip_id, 'wp_travel_minimum_partial_payout_percent', true );
			$order_items    = get_post_meta( $booking_id, 'order_items_data', true );
			$trip_price     = ! empty( $order_items[ $first_key ]['trip_price'] ) ? $order_items[ $first_key ]['trip_price'] : '';

			// Need to update $use_global if cart has multiple items.
			if ( is_array( $order_items ) && count( $order_items ) > 1 ) {
				$use_global = true;
			}

			$partial_payments = ! empty( $settings['minimum_partial_payout'] ) ? $settings['minimum_partial_payout'] : array();
			$payout_data      = array(
				'trip_id'          => $trip_id,
				'trip_price'       => $trip_price,
				'partial_payments' => $partial_payments,
				'use_global'       => $use_global,
				'custom_payouts'   => $custom_payouts,
			);
			update_post_meta( $booking_id, 'wp_travel_partial_payment_array_structure', $payout_data );
		}

		/**
		 * Sets the new due amount to the user dashboard.
		 *
		 * @param string $due_amount
		 * @return string $due_amount
		 */
		public static function due_amount( $due_amount ) {
			
			if ( empty( $due_amount ) ) {
				return $due_amount;
			}

			$booking_id           = isset( $_GET['detail_id'] ) && ! empty( $_GET['detail_id'] ) ? sanitize_text_field( wp_unslash( $_GET['detail_id'] ) ) : 0;
			$partial_payment_data = get_post_meta( $booking_id, 'wp_travel_partial_payment_array_structure', true );

			$partial_payments = ! empty( $partial_payment_data['partial_payments'] ) ? $partial_payment_data['partial_payments'] : array();
			$use_global       = isset( $partial_payment_data['use_global'] ) ? $partial_payment_data['use_global'] : false;
			$custom_payouts   = ! empty( $partial_payment_data['custom_payouts'] ) ? $partial_payment_data['custom_payouts'] : array();
			$trip_price       = ! empty( $partial_payment_data['trip_price'] ) ? $partial_payment_data['trip_price'] : array();
			$trip_id          = ! empty( $partial_payment_data['trip_id'] ) ? $partial_payment_data['trip_id'] : array();
			$payment_ids      = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
			$payment_ids      = ! is_array( $payment_ids ) ? array( $payment_ids ) : $payment_ids;
			// $count_payments   = count( $payment_ids );
			$payment_data       = wptravel_payment_data( $booking_id );
			$count_payments     = count( $payment_data ); // count this instead of payment_ids to get real number of payment made.
			if ( function_exists( 'wptravel_booking_data' ) ) {
				$details = wptravel_booking_data( $booking_id );
			} else {
				$details = wp_travel_booking_data( $booking_id );
			}
			$is_custom_valid = WP_Travel_Custom_Multiple_Partial_Payment::is_custom_payout_valid( $trip_id );
			$total_amount    = isset( $details['total'] ) && ! empty( $details['total'] ) ? (float) $details['total'] : 0;
			$booking_option  = $details['booking_option'];

			if ( ! $use_global && $is_custom_valid ) {
				$partial_payments = $custom_payouts;
			}

			/**
			 * @since WP Travel Partial Payment 1.2.0, $trip_price has been changed to $total_amount to fix the tax issue.
			 *
			 * === Snippet of old code ===
				if ( '0.00' !== $due_amount ) {
					if ( is_array( $partial_payments ) && count( $partial_payments ) > 0 ) {
						$partial_payment = isset( $partial_payments[ $count_payments ] ) && ! empty( $partial_payments[ $count_payments ] ) ? (float) $partial_payments[ $count_payments ] : '';
						$due_amount      = ! empty( $partial_payment ) && ! empty( $trip_price ) ? ( $trip_price * $partial_payment ) / 100 : $due_amount;
						$due_amount      = wptravel_get_formated_price( $due_amount );
					}
				}
			 * === Snippet of old code ===
			 */
			if ( '0.00' !== $due_amount ) {
				if ( 'booking_only' === $booking_option ) {
					// To support partial payment in case of booking only.
					// $count_payments = 0; 
				} 
				// Calculate Due amount.
				if ( wptravel_get_settings()['partial_payment_amount'] == 'no' && is_array( $partial_payments ) && count( $partial_payments ) > 0 ) {
					$partial_payment = isset( $partial_payments[ $count_payments ] ) && ! empty( $partial_payments[ $count_payments ] ) ? (float) $partial_payments[ $count_payments ] : '';
					$due_amount      = ! empty( $partial_payment ) && ! empty( $total_amount ) ? ( $total_amount * $partial_payment ) / 100 : $due_amount;
					$due_amount      = wptravel_get_formated_price( $due_amount );
				}
			}

			return $due_amount;
		}

		/**
		 * Displays information for the current installment.
		 *
		 * @param array $payment_fields
		 * @param int   $booking_id
		 * @return array $payment_fields
		 */
		public static function dashboard_installment_information( $payment_fields, $booking_id ) {
			if ( empty( $payment_fields ) || empty( $booking_id ) ) {
				return $payment_fields;
			}

			$partial_payment_data = get_post_meta( $booking_id, 'wp_travel_partial_payment_array_structure', true );
			$global_payouts       = $partial_payment_data['partial_payments'];
			$custom_payouts       = ! empty( $partial_payment_data['custom_payouts'] ) ? $partial_payment_data['custom_payouts'] : array();

			// Calculate merged custom payout. This will help to count number of partial payout if custom partial data are not added yet.
			$result = array();
			foreach ( $global_payouts as $key => $val ) {
				$new_val        = isset( $custom_payouts[ $key ] ) && $custom_payouts[ $key ] ? $custom_payouts[ $key ] : $val; // Get the values from the other array
				$result[ $key ] = $new_val;
			}
			$custom_payouts = $result;

			if ( function_exists( 'wptravel_booking_data' ) ) {
				$details = wptravel_booking_data( $booking_id );
			} else {
				$details = wp_travel_booking_data( $booking_id );
			}
			$due_amount         = ! empty( $details['due_amount'] ) ? $details['due_amount'] : '';
			$partial_amount = wptravel_get_formated_price_currency( wptravel_get_settings()['partial_amount'], false, '', $booking_id );

			$partial_payments   = ! empty( $partial_payment_data['use_global'] ) ? $partial_payment_data['partial_payments'] : $custom_payouts;
			$payment_ids        = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
			$payment_ids        = ! is_array( $payment_ids ) ? array( $payment_ids ) : $payment_ids;
			$payment_data       = wptravel_payment_data( $booking_id );
			$count_payments     = count( $payment_data ); // count this instead of payment_ids to get real number of payment made.
			$count_installments = is_array( $partial_payments ) ? count( $partial_payments ) : 0;
			if ( isset( $partial_payments[0] ) && (float) $partial_payments[0] >= 100 ) {
				// override no of installment in case of 1st partial payment as 100%.
				$count_installments = 1;
			}
			$booking_option     = $details['booking_option'];
			if ( '0.00' !== $due_amount && $count_installments > 0 ) {
				if ( 'booking_only' === $booking_option ) {
					// To support partial payment in case of booking only.
					// $count_payments = 0; 
				} 
				$payout_percent = isset( $partial_payments[ $count_payments ] ) ? $partial_payments[ $count_payments ] : 0;
				/* Translators: %d installment info. */
				if ( wptravel_get_settings()['partial_payment_amount'] == 'yes'){
					$payment_fields['amount_info']['label'] = __( 'Remanning Payment Amount ', 'wp-travel-pro' );
				}else{
					$payment_fields['amount_info']['label'] = __( 'Installment ', 'wp-travel-pro' ) .  ( $count_payments + 1 ) . __( ' out of ', 'wp-travel-pro' ) . $count_installments . ' ( ' .  $payout_percent . '% )';
				}
				
			}
			return $payment_fields;
		}
	}
}
WP_Travel_Partial_Payment_Frontend_Process::init();
