<?php
/**
 * This file has required code for the multiple partial payment email notification cycle.
 */

if ( ! class_exists( 'WP_Travel_Partial_Payment_Reminder_Email' ) ) {

	/**
	 * Class for the reminder email.
	 */
	class WP_Travel_Partial_Payment_Reminder_Email {

		/**
		 * Initialize class.
		 *
		 * @return void
		 */
		public static function init() {
			// add_action( 'cron_schedules', array( __CLASS__, 'add_cron_recurrence_interval' ) );
			self::set_cron_job();
			add_action( 'wp_travel_partial_payment_reminder_email_cycle', array( __CLASS__, 'trigger_cron_job' ) );
			self::clear_cron_job();
		}

		/**
		 * Add the custom cron intervals.
		 *
		 * @param array $schedules
		 * @return void
		 */
		public static function add_cron_recurrence_interval( $schedules ) {
			// Only for testing purpose.
			$schedules['every_five_min'] = array(
				'interval' => 300, // In secs.
				'display'  => 'Every 5 minute',
			);
			// Only for testing purpose.

			$schedules['once_in_four_days']    = array(
				'interval' => 345600,
				'display'  => 'Once in 4 days',
			);
			$schedules['once_in_seven_days']   = array(
				'interval' => 604800,
				'display'  => 'Once in 7 days',
			);
			$schedules['once_in_fifteen_days'] = array(
				'interval' => 1317600,
				'display'  => 'Once in 15 days',
			);
			return $schedules;
		}

		/**
		 * If cron job is not already scheduled then scheduled one and start job in selected schedules.
		 */
		public static function set_cron_job() {
			// wt_twicedaily this cron is run twice in a day.
			if ( ! wp_next_scheduled( 'wp_travel_partial_payment_reminder_email_cycle' ) ) {
				wp_schedule_event( time(), 'wt_twicedaily', 'wp_travel_partial_payment_reminder_email_cycle' );
			}
		}

		/**
		 * Clear cron job on plugin deactivation.
		 *
		 * @return void
		 */
		public static function _clear_cron_job() {
			wp_clear_scheduled_hook( 'wp_travel_partial_payment_reminder_email_cycle' );
		}

		/**
		 * Calls the plugin deactivation hook.
		 *
		 * @return void
		 */
		public static function clear_cron_job() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$send_notification = ! empty( $settings['remaining_partial_payment_reminder']['send_remaining_payment_notification'] ) ? $settings['remaining_partial_payment_reminder']['send_remaining_payment_notification'] : '';
			if ( 'yes' !== $send_notification ) {
				wp_clear_scheduled_hook( 'wp_travel_partial_payment_reminder_email_cycle' );
			}
			if ( defined( 'WP_TRAVEL_PARTIAL_PAYMENT_PLUGIN_FILE' ) ) {
				$plugin_main_file_path = WP_TRAVEL_PARTIAL_PAYMENT_PLUGIN_FILE;
			} elseif ( defined( 'WP_TRAVEL_PRO_PLUGIN_FILE' ) ) {
				$plugin_main_file_path = WP_TRAVEL_PRO_PLUGIN_FILE;
			}
			register_deactivation_hook( $plugin_main_file_path, array( __CLASS__, '_clear_cron_job' ) );
		}

		/**
		 * Triggers the cron job according to the schedule.
		 *
		 * @return void
		 */
		public static function trigger_cron_job() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$email_cycle            = ! empty( $settings['remaining_partial_payment_reminder']['email_cycle'] ) ? $settings['remaining_partial_payment_reminder']['email_cycle'] : array( '1' ); // send email before one trip day.
			$subject                = ! empty( $settings['remaining_partial_payment_reminder']['reminder_title'] ) ? $settings['remaining_partial_payment_reminder']['reminder_title'] : 'Remaining Payment';
			$reminder_email_content = ! empty( $settings['remaining_partial_payment_reminder']['email_content'] ) ? $settings['remaining_partial_payment_reminder']['email_content'] : wp_travel_partial_payment_reminder_email_content();
			$send_notification      = ! empty( $settings['remaining_partial_payment_reminder']['send_remaining_payment_notification'] ) ? $settings['remaining_partial_payment_reminder']['send_remaining_payment_notification'] : '';

			$args = array(
				'post_type'      => 'itinerary-booking',
				'post_status'    => 'publish',
				'posts_per_page' => -1,
				'meta_query'     => array(
					array(
						'key'     => 'stop_reminder_notification',
						'compare' => 'NOT EXISTS',
					),
				),
			);

			// Fixes for Previous email cycle value as strings.
			if ( ! is_array( $email_cycle ) ) {
				$cycle       = (int) $email_cycle ? (int) $email_cycle : 1; // Send email before 1 | 5 | 15 day of trip start.
				$email_cycle = array( $cycle );
			}

			if ( 'yes' === $send_notification ) {
				// The Query.
				$the_query = new WP_Query( $args );
				$the_posts = ! empty( $the_query->posts ) ? $the_query->posts : '';

				if ( is_array( $the_posts ) || is_object( $the_posts ) && count( $the_posts ) > 0 ) {
					foreach ( $the_posts as $the_post ) {
						$booking_id = $the_post->ID;
						if ( function_exists( 'wptravel_booking_data' ) ) {
							$details = wptravel_booking_data( $booking_id );
						} else {
							$details = wp_travel_booking_data( $booking_id );
						}

						$due_amount  = isset( $details['due_amount'] ) && ! empty( $details['due_amount'] ) ? $details['due_amount'] : 0;
						$trip_id     = get_post_meta( $booking_id, 'wp_travel_post_id', true );
						$first_names = get_post_meta( $booking_id, 'wp_travel_fname_traveller', true );
						$last_names  = get_post_meta( $booking_id, 'wp_travel_lname_traveller', true );
						$emails      = get_post_meta( $booking_id, 'wp_travel_email_traveller', true );
						$trip_date   = get_post_meta( $booking_id, 'wp_travel_arrival_date', true );

						if ( $due_amount > 0 ) {
							$today = gmdate( 'Y-m-d H:i:s' );
							foreach ( $email_cycle as $email_cycle_day ) {
								$compare_date = gmdate( 'Y-m-d H:i:s', strtotime( $trip_date . ' -' . $email_cycle_day . ' day' ) ); // deduct number of days from current date.
								$cycle_meta_name          = sprintf( 'sent_reminder_before_%s_day', $email_cycle_day );
								$sent_current_cycle_email = get_post_meta( $booking_id, $cycle_meta_name, true );
								if ( $today >= $compare_date && 'yes' !== $sent_current_cycle_email ) {
									if ( is_array( $first_names ) && ! empty( $first_names ) ) {
										foreach ( $first_names as $first_key => $first_name ) {
											$booking_edit_link = site_url( '/wp-travel-dashboard/' ) . '?detail_id=' . $booking_id . '#bookings';
											$email_tags        = array(
												'{itinerary_title}' => get_the_title( $trip_id ),
												'{booking_id}'    => $booking_id,
												'{booking_edit_link}' => $booking_edit_link,
												'{customer_name}' => $first_name[0] . ' ' . $last_names[ $first_key ][0],
											);

											$to       = ! empty( $emails[ $first_key ][0] ) ? $emails[ $first_key ][0] : '';
											$message  = str_replace( array_keys( $email_tags ), $email_tags, $reminder_email_content );
											$headers  = 'From: ' . get_option( 'admin_email' ) . "\r\n";
											$headers .= 'Reply-To: no-reply@' . site_url() . "\r\n";
											$headers .= "MIME-Version: 1.0\r\n";
											$headers .= "Content-Type: text/html; charset=ISO-8859-1\r\n";
											if ( ! empty( $to ) ) {
												wp_mail( $to, $subject, $message, $headers );
												update_post_meta( $booking_id, $cycle_meta_name, 'yes' );
											}
										}
									}
								}
							}
						} else {
							update_post_meta( $booking_id, 'stop_reminder_notification', 'yes' );
						}
					}
				}
				wp_reset_postdata();
			}
		}
	}
}
WP_Travel_Partial_Payment_Reminder_Email::init();
