<?php
/**
 * PayHere Payment Class.
 *
 * @package WP_Travel_PayHere_Checkout_Core
 */

if ( ! class_exists( 'WP_Travel_PayHere_Payment' ) ) :

	/**
	 * PayHere Payment Handler.
	 */
	class WP_Travel_PayHere_Payment {

		/**
		 * Defaults Args.
		 *
		 * @var $defaults
		 */
		private $defaults_vars;

		/**
		 * Holds Booking Id.
		 *
		 * @var $booking_id
		 */
		private $booking_id;

		/**
		 * Complete Partial Payment.
		 *
		 * @var $is_complete_partial_payment
		 */
		private $is_complete_partial_payment;

		/**
		 * Constructor.
		 *
		 * @param int  $booking_id Booking ID.
		 * @param bool $complete_partial_payment Is Complete Partial Payment.
		 */
		public function __construct( $booking_id = null, $complete_partial_payment = false ) {
			$this->booking_id                  = $booking_id;
			$this->is_complete_partial_payment = $complete_partial_payment;
			$this->defaults_vars               = array();
		}

		/**
		 * Get Return URL.
		 */
		public function get_return_url() {
			$trip_id = get_post_meta( (int) $this->booking_id, 'wp_travel_post_id', true );
			$url     = '';
			if ( function_exists( 'wp_travel_thankyou_page_url' ) ) {
				$url = wptravel_thankyou_page_url( (int) $trip_id );
			}
			return $url;
		}

		/**
		 * Merchant's Details Vars.
		 */
		private function get_merchant_details() {
			$args     = array();
			$settings = $this->get_payment_settings();
			extract( $settings ); // $merchant_id, $merchant_key, $passphrase
			$return_url = $this->get_return_url();
			// Merchant's Details.
			$args['merchant_id'] = $merchant_id;
			$args['secret']      = $merchant_secret;
			$args['order_id']    = $this->booking_id;
			$args['return_url']  = add_query_arg(
				array(
					'booking_id' => $this->booking_id,
					'booked'     => true,
					'status'     => 'success',
					'order_id'   => $this->booking_id,
				),
				$return_url
			);
			$args['cancel_url']  = add_query_arg(
				array(
					'booking_id' => $this->booking_id,
					'booked'     => true,
					'status'     => 'cancel',
				),
				$return_url
			);
			$args['notify_url']  = esc_url( add_query_arg( 'payhere_listener', 'IPN', home_url( 'index.php' ) ) );
			return $args;
		}

		/**
		 * Buyer's Details Vars.
		 */
		private function get_buyer_details() {
			$booking_data = get_post_meta( (int) $this->booking_id, 'order_data', true );

			$args = array();
			$key  = $this->get_form_index_key();
			if ( $key ) :
				// Buyer Details.
				$args['first_name'] = isset( $booking_data['wp_travel_fname_traveller'][ $key ][0] ) ? $booking_data['wp_travel_fname_traveller'][ $key ][0] : '';
				$args['last_name']  = isset( $booking_data['wp_travel_lname_traveller'][ $key ][0] ) ? $booking_data['wp_travel_lname_traveller'][ $key ][0] : '';
				$args['email']      = isset( $booking_data['wp_travel_email_traveller'][ $key ][0] ) ? $booking_data['wp_travel_email_traveller'][ $key ][0] : '';
				$args['phone']      = isset( $booking_data['wp_travel_phone_traveller'][ $key ][0] ) ? $booking_data['wp_travel_phone_traveller'][ $key ][0] : '';
				$args['address']    = isset( $booking_data['wp_travel_address'] ) ? $booking_data['wp_travel_address'] : '';
				$args['city']       = isset( $booking_data['billing_city'] ) ? $booking_data['billing_city'] : '';
				$args['country']    = isset( $booking_data['wp_travel_country']) ? $booking_data['wp_travel_country']: '';
			endif;
			return $args;
		}

		private function get_hash() {
			$settings    = $this->get_payment_settings();
			$merchant    = $this->get_merchant_details();
			$arg         = array();
			$trans       = $this->get_transaction_details();
			$arg['hash'] = strtoupper( md5( $merchant['merchant_id'] . $trans['order_id'] . $trans['amount'] . $trans['currency'] ) . strtoupper( md5( $merchant['secret'] ) ) );
			return $arg;
		}

		/**
		 * Transaction Details.
		 */
		private function get_transaction_details() {
			global $wt_cart;
			$total_price = $wt_cart->get_total();

			$payment_mode = isset( $_POST['wp_travel_payment_mode'] ) ? $_POST['wp_travel_payment_mode'] : 'full';

			$complete_partial_payment = $this->is_complete_partial_payment;

			$arg     = array();
			$trip_id = get_post_meta( (int) $this->booking_id, 'wp_travel_post_id', true );
			
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			if ( $complete_partial_payment ) {
				$amount        = isset( $_POST['amount'] ) ? $_POST['amount'] : 0;
				$arg['amount'] = $amount;
			} else {
				if ( 'partial' === $payment_mode ) {
					$arg['amount'] = $total_price['total_partial'];
				} else {
					$arg['amount'] = $total_price['total'];
				}
			}
			$arg['currency'] = isset( $settings['currency'] ) ? $settings['currency'] : '';
			$arg['items']    = get_the_title( (int) $trip_id );

			return $arg;
		}

		public function get_payment_api_url( $ssl_check = false ) {
			return $this->get_payment_host();
		}

		/**
		 * Payment Host.
		 *
		 * @param boolean $ssl_check Test SSL.
		 * @return string
		 */
		public function get_payment_host( $ssl_check = false ) {
			if ( is_ssl() || ! $ssl_check ) {
				$protocol = 'https://';
			} else {
				$protocol = 'http://';
			}

			if ( function_exists( 'wptravel_test_mode' ) ) {
				$test_mode = wptravel_test_mode();
			} else {
				$test_mode = wp_travel_test_mode();
			}
			if ( $test_mode ) {
				$host = $protocol . 'sandbox.payhere.lk/pay/checkout';
			} else {
				$host = $protocol . 'www.payhere.lk/pay/checkout';
			}

			return $host;
		}

		/**
		 * Checkout Form Index Key.
		 */
		private function get_form_index_key() {
			$order_details = get_post_meta( (int) $this->booking_id, 'order_items_data', true ); // Multiple Trips.

			$index = is_array( $order_details ) && count( $order_details ) > 0 ? array_keys( $order_details )[0] : null;
			return $index;
		}

		public function get_payment_args() {
			$arg1 = $this->get_merchant_details();
			$arg2 = $this->get_buyer_details();
			$arg3 = $this->get_transaction_details();
			$arg4 = $this->get_custom_args();
			// $arg5 = $this->get_hash();
			return array_merge( $arg1, $arg2, $arg3, $arg4 );
		}

		public function get_custom_args() {
			$arg             = array();
			$arg['custom_1'] = (int) $this->booking_id;
			return $arg;
		}

		public function get_payment_form() {
			$params   = array(
				'merchant_id',
				'return_url',
				'cancel_url',
				'notify_url',
				'first_name',
				'last_name',
				'email',
				'phone',
				'address',
				'city',
				'custom_1',
				'country',
				'order_id',
				'items',
				'currency',
				'amount',
			);
			$args     = $this->get_payment_args();
			$url_keys = array( 'return_url', 'cancel_url', 'notify_url' );
			$api_url  = $this->get_payment_api_url();
			?>
			<form id="wp_travel_payhere_payment_form" method="post" action="<?php echo esc_url( $api_url ); ?>">
				<h3><?php esc_html_e( 'You will redirect shortly ...', 'wp-travel-pro' ); ?></h3>
				<?php
				foreach ( $params as $key ) :
					if ( ! isset( $args[ $key ] ) ) {
						continue;
					}
					if ( array_key_exists( $key, $url_keys ) ) {
						echo '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_url( $args[ $key ] ) . '">';
						continue;
					}
					echo '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_attr( $args[ $key ] ) . '">';
				endforeach;
				?>
				<input type="submit" value="Pay"/>
			</form>
			<script>
				var form = document.getElementById( 'wp_travel_payhere_payment_form' );
				form.querySelector('input[type=submit]').style.display = 'none';
				form.submit();
			</script>
			<?php
		}

		/**
		 * Payment Settings.
		 */
		private function get_payment_settings() {
			$settings_instance = WP_Travel_PayHere_Settings::instance();
			$settings          = $settings_instance->get_settings();
			return $settings;
		}
	}

endif;
