<?php
/**
 * WP Travel Paypal Express Checkout Core Class.
 *
 * @package wp-travel-paypal-express-checkout-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
// WP Travel Paypal Express Checkout core.
if ( ! class_exists( 'WP_Travel_Paypal_Express_Checkout_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Paypal_Express_Checkout_Core {

		const WP_TRAVEL_PAYPAL_EXPRESS_CHECKOUT_HANDLE = 'wp_travel_paypal_express_checkout_';
		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Paypal Express Checkout Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Paypal_Express_Checkout_Core Instance.
		 * Ensures only one instance of WP_Travel_Paypal_Express_Checkout_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Paypal_Express_Checkout_Core()
		 * @return WP_Travel_Paypal_Express_Checkout_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			self::includes();

			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Paypal_Express_Checkout_Core', 'frontend_assets' ), 30 );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Paypal_Express_Checkout_Core', 'admin_assets' ), 30 );

			// Payment Gateway list.
			add_filter( 'wp_travel_payment_gateway_lists', 'wp_travel_gateway_paypal_express_checkout' );
			add_filter( 'wp_travel_premium_addons_list', 'wp_travel_addons_paypal_express_checkout' );
			if ( self::is_enabled() ) {
				add_filter( 'wptravel_frontend_data', 'wp_travel_add_vars_paypal_express_checkout', 10, 2 );
			}

			if ( self::uses_paypal_express_checkout() ) {
				// This will call after payement completed.
				add_action( 'wp_travel_after_frontend_booking_save', array( 'WP_Travel_Paypal_Express_Checkout_Core', 'set_up_payment_options' ), 30 );
				add_action( 'wp_travel_after_partial_payment_complete', array( 'WP_Travel_Paypal_Express_Checkout_Core', 'set_up_partial_payment' ), 30 );
			}

			isset( $_SESSION['used-express-checkout'] ) && $_SESSION['used-express-checkout'] && add_filter( 'wp_travel_booked_message', array( 'WP_Travel_Paypal_Express_Checkout_Core', 'booking_message' ), 30 );

		}

		/**
		 * Determine if booking used express checkout.
		 */
		private static function uses_paypal_express_checkout() {
			$settings       = wptravel_get_settings();
			if( isset( $settings['selected_booking_option'] ) &&  count( $settings['selected_booking_option'] ) == 1 && $settings['selected_booking_option'][0] = 'booking-with-payment' ){ 
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'express_checkout' === $_REQUEST['wp_travel_payment_gateway'];
			}else{
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_booking_option', $_REQUEST ) && 'booking_with_payment' === $_REQUEST['wp_travel_booking_option'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'express_checkout' === $_REQUEST['wp_travel_payment_gateway'];
			}
			
		}

		/**
		 * Alter the booking travel message as payment has already been done.
		 *
		 * @param string $message Set trip booked message.
		 * @return string
		 */
		public static function booking_message( $message ) {

			unset( $_SESSION['used-express-checkout'] );

			$message = esc_html__( "We've received your booking and payment details. We'll contact you soon.", 'wp-travel-pro' );

			return $message;

		}

		/**
		 * Determine if checkout is enabled.
		 */
		private static function is_enabled() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			return array_key_exists( 'payment_option_express_checkout', $settings ) && 'yes' === $settings['payment_option_express_checkout'];
		}

		/**
		 * Determing is checkout is disabled.
		 */
		private static function is_disabled() {
			return ! self::is_enabled();
		}

		/**
		 * Frontend assets.
		 */
		public static function frontend_assets() {
			
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			if ( function_exists( 'wptravel_can_load_payment_scripts' ) ) {
				$load_payment_script = wptravel_can_load_payment_scripts();
			} else {
				$load_payment_script = wp_travel_can_load_payment_scripts();
			}


			if ( function_exists( 'wptravel_test_mode' ) ) {
				$test_mode = wptravel_test_mode();
			} else {
				$test_mode = wp_travel_test_mode();
			}

			if( $test_mode ){
				$client_id = isset( $settings['wp_travel_express_checkout_sand_box']['client_id'] ) ? $settings['wp_travel_express_checkout_sand_box']['client_id'] : '';
			}else{
				$client_id = isset( $settings['wp_travel_express_checkout_production']['client_id'] ) ? $settings['wp_travel_express_checkout_production']['client_id'] : '';
			}

			

			if ( $load_payment_script && self::is_enabled() && isset( $settings['enable_one_page_booking'] ) && $settings['enable_one_page_booking'] == "" ) {
				$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

				if( apply_filters( 'wp_travel_use_paypal_sdk_v2', true ) == false ){
					wp_enqueue_script( 'wp-travel-paypal-express-checkout-server', 'https://www.paypalobjects.com/api/checkout.js', array(), '0.0.1', true );
					wp_enqueue_script( 'wp-travel-paypal-express-checkout', self::$plugin_url . 'assets/js/wp-travel-paypal-express-checkout.js', array( 'wp-travel-paypal-express-checkout-server' ), '0.0.1', true );
				}else{
					wp_enqueue_script( 'wp-travel-paypal-express-checkout-server', 'https://www.paypal.com/sdk/js?client-id='.$client_id.'&currency='.$settings['currency'], array(), null, true );
					wp_enqueue_script( 'wp-travel-paypal-express-checkout', self::$plugin_url . 'assets/js/wp-travel-paypal-express-checkout-v2.js', array( 'wp-travel-paypal-express-checkout-server' ), '0.0.1', true );
				}
				
			}
			global $post;
			// wp-travel-authorize-net-checkout-frontend
			if ( get_post_type() == 'itineraries' && isset( $settings['enable_one_page_booking'] ) && $settings['enable_one_page_booking'] == 1 ) {

				wp_enqueue_script( 'wp-travel-paypal-express-checkout-server', 'https://www.paypal.com/sdk/js?client-id='.$client_id.'&currency='.$settings['currency'], array(), null, true );
				$depend = array( 'wp-travel-frontend-booking-widget', 'wp-travel-paypal-express-frontend-server' );
				wp_enqueue_script( 'wp-travel-express-frontend-script', self::$plugin_url . 'app/build/wp-travel-frontend-booking.js', array(), '0.0.1', true );
			
				
				// wp_enqueue_script( 'wp-travel-express-frontend-script' );
			}

			if( apply_filters( 'wp_travel_enable_paypal_express_js', false ) == true ){
				if( apply_filters( 'wp_travel_use_paypal_sdk_v2', true ) == false ){
					wp_enqueue_script( 'wp-travel-paypal-express-checkout-server', 'https://www.paypalobjects.com/api/checkout.js', array(), '0.0.1', true );
					wp_enqueue_script( 'wp-travel-paypal-express-checkout', self::$plugin_url . 'assets/js/wp-travel-paypal-express-checkout.js', array( 'wp-travel-paypal-express-checkout-server' ), '0.0.1', true );
				}else{
					wp_enqueue_script( 'wp-travel-paypal-express-checkout-server', 'https://www.paypal.com/sdk/js?client-id='.$client_id.'&currency='.$settings['currency'], array(), null, true );
					wp_enqueue_script( 'wp-travel-paypal-express-checkout', self::$plugin_url . 'assets/js/wp-travel-paypal-express-checkout-v2.js', array( 'wp-travel-paypal-express-checkout-server' ), '0.0.1', true );
				}
			}
		}

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {

			$suffix         = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$screen         = get_current_screen();
			$allowed_screen = array( 'edit-itineraries', WP_TRAVEL_POST_TYPE, 'itineraries_page_settings', 'itinerary-booking', 'itinerary-booking_page_settings' );
			$screen         = get_current_screen();
			if (  method_exists( 'WP_Travel',  'is_page' )  ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_PAYPAL_EXPRESS_CHECKOUT_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			include sprintf( '%sinc/options.php', self::$abspath );
			include sprintf( '%sinc/functions.php', self::$abspath );
			if ( self::is_request( 'admin' ) ) {
				include sprintf( '%sinc/admin/settings.php', self::$abspath );
			}
		}

		/**
		 * Sets up payment options
		 *
		 * @param string $booking_id ID of booking.
		 * @return void
		 */
		public static function set_up_payment_options( $booking_id ) {

			if ( self::is_disabled() ) {
				return;
			}

			if ( ! self::uses_paypal_express_checkout() ) {
				return;
			}

			if ( ! $booking_id ) {
				return;
			}

			do_action( 'wt_before_payment_process', $booking_id );

			$json = sanitize_text_field( wp_unslash( $_REQUEST['payment_details'] ) );

			$detail = json_decode( $json );
			if( apply_filters( 'wp_travel_use_paypal_sdk_v2', true ) == false ){
				$amount = array_sum( array_map( 'wp_travel_sum_amounts', $detail->transactions ) );
			}else{
				$amount = array_sum( array_map( 'wp_travel_sum_amounts', $detail->purchase_units ) );
			}

			$payment_id = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
			$payment_method = 'express_checkout';
			update_post_meta( $payment_id, 'wp_travel_payment_gateway', $payment_method );

			wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_method ), $payment_id );

			$_SESSION['used-express-checkout'] = true;

			do_action( 'wp_travel_after_successful_payment', $booking_id );

		}

		/**
		 * Sets up payment options
		 *
		 * @return void
		 */
		public static function set_up_partial_payment() {

			if ( self::is_disabled() ) {
				return;
			}

			if ( ! self::uses_paypal_express_checkout() ) {
				return;
			}

			if ( ! isset( $_POST['complete_partial_payment'] ) ) {
				return;
			}

			if ( ! isset( $_POST['wp_travel_booking_id'] ) ) {
				return;
			}
			$payment_gateway = sanitize_text_field( wp_unslash( $_POST['wp_travel_payment_gateway'] ) );
			$booking_id      = sanitize_text_field( wp_unslash( $_POST['wp_travel_booking_id'] ) );
			// get previous payment ids.
			$payment_id = get_post_meta( $booking_id, 'wp_travel_payment_id', true );

			/**
			 * This will return the current payment id to insert payment meta for the current payment.
			 * 
			 * @hooked wptravel_update_payment_post from partial payment which will also update booking option [booking_only | booking_with_payment ]
			 *
			 * @since 4.0.2
			 */
			$new_payment_id = apply_filters( 'wptravel_before_insert_partial_payment', $payment_id, $booking_id, $payment_gateway );

			$json = sanitize_text_field( wp_unslash( $_REQUEST['payment_details'] ) );

			$detail = json_decode( $json );
			if( apply_filters( 'wp_travel_use_paypal_sdk_v2', true ) == false ){
				$amount = array_sum( array_map( 'wp_travel_sum_amounts', $detail->transactions ) );
			}else{
				$amount = array_sum( array_map( 'wp_travel_sum_amounts', $detail->purchase_units ) );
			}

			if ( $amount ) {
				wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_gateway ), $new_payment_id );

				$_SESSION['used-express-checkout'] = true;
			}
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}
	}
endif;
