<?php
/**
 * Modules core file.
 *
 * @package wp-travel-paystack-checkout-core
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Custom Filters Core.
if ( ! class_exists( 'WP_Travel_Paystack_Checkout_Core' ) ) :
	/**
	 * WP Travel Paystack Checkout Core.
	 */
	class WP_Travel_Paystack_Checkout_Core {

		const WP_TRAVEL_PAYSTACK_CHECKOUT_HANDLE = 'wp_travel_paystack_checkout_';

		/**
		 * Absolute path to core
		 *
		 * @var string
		 */
		protected static $abspath;

		/**
		 * Undocumented variable
		 *
		 * @var [type]
		 */
		protected static $plugin_file;

		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Paystack Checkout Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Paystack_Checkout_Core Instance.
		 * Ensures only one instance of WP_Travel_Paystack_Checkout_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Paystack_Checkout_Core()
		 * @return WP_Travel_Paystack_Checkout_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param Array $params Core class init paramerters.
		 */
		public static function init( $params ) {
			self::$abspath     = $params['abspath'] . 'inc/modules/paystack-checkout-core/';
			self::$plugin_file = __FILE__;
			self::$version     = $params['version'];
			self::includes();
			self::init_hooks();
		}

		/**
		 * Includes required files.
		 */
		public static function includes() {
			include_once self::$abspath . 'inc/admin/settings.php';
			include_once self::$abspath . 'inc/functions.php';
		}

		/**
		 * Init Hooks.
		 */
		public static function init_hooks() {
			add_action( 'wp_enqueue_scripts', array( __CLASS__, 'assets' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'admin_assets' ), 20 );

			// Payment Gateway list.
			add_filter( 'wp_travel_payment_gateway_lists', 'wp_travel_gateway_paystack' );
			add_filter( 'wp_travel_premium_addons_list', 'wp_travel_paystack_addons' );

			if ( self::is_enabled() ) {
				add_filter( 'wptravel_frontend_data', 'wp_travel_paystack_add_vars', 10, 2 );
			}

			if ( self::uses_paystack_checkout() ) {
				add_action( 'wp_travel_after_frontend_booking_save', array( 'WP_Travel_Paystack_Checkout_Core', 'set_up_payment_options' ), 20, 1 );
				// add_action( 'wp_travel_after_partial_payment_complete', array( 'WP_Travel_Paystack_Checkout_Core', 'set_up_partial_payment' ), 20 );
			}

			// General Notices.
			add_filter( 'wp_travel_display_general_admin_notices', array( __CLASS__, 'wp_travel_display_paystack_notices' ), 20 );
			add_action( 'wp_travel_general_admin_notice', array( __CLASS__, 'wp_travel_paystack_notices' ), 20 );

			if ( isset( $_SESSION['used-paystack'] ) && $_SESSION['used-paystack'] ) {
				add_filter( 'wp_travel_booked_message', array( __CLASS__, 'booking_message' ), 20 );
			}
		}

		public static function set_up_payment_options( $booking_id ) {
			if ( self::is_disabled() ) {
				return;
			}

			if ( ! self::uses_paystack_checkout() ) {
				return;
			}

			if ( ! $booking_id ) {
				return;
			}
			if ( ! isset( $_POST['wp_travel_book_now'] ) ) {
				return;
			}

			do_action( 'wt_before_payment_process', $booking_id );

			$json   = sanitize_text_field( wp_unslash( $_REQUEST['payment_details'] ) );
			$detail = json_decode( $json );
			$amount = isset( $detail->amount ) ? $detail->amount : 0;

			if ( $amount ) {
				$amount = number_format( $amount, 2, '.', '' );
			}
			wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, '_paystack_args' );
			$payment_id                = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
			$_SESSION['used-paystack'] = true;
			do_action( 'wp_travel_after_successful_payment', $booking_id );
		}

		public static function set_up_partial_payment() {
			if ( self::is_disabled() ) {
				return;
			}

			if ( ! self::uses_paystack_checkout() ) {
				return;
			}

			if ( ! isset( $_POST['complete_partial_payment'] ) ) {
				return;
			}

			if ( ! isset( $_POST['wp_travel_booking_id'] ) ) {
				return;
			}

			$payment_gateway = sanitize_text_field( wp_unslash( $_POST['wp_travel_payment_gateway'] ) );
			$booking_id      = sanitize_text_field( wp_unslash( $_POST['wp_travel_booking_id'] ) );
			// get previous payment ids.
			$payment_id      = get_post_meta( $booking_id, 'wp_travel_payment_id', true );

			/**
			 * This will return the current payment id to insert payment meta for the current payment.
			 * 
			 * @hooked wptravel_update_payment_post from partial payment which will also update booking option [booking_only | booking_with_payment ]
			 *
			 * @since 4.0.2
			 */
			$new_payment_id = apply_filters( 'wptravel_before_insert_partial_payment', $payment_id, $booking_id, $payment_gateway );

			$json = sanitize_text_field( wp_unslash( $_POST['payment_details'] ) );

			$detail = json_decode( $json );
			$amount = isset( $detail->amount ) ? $detail->amount : 0;
			if ( $amount ) {
				$amount = number_format( $amount / 100, 2, '.', '' );
				wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_gateway ), $new_payment_id );

				$_SESSION['used-paystack'] = true;
			}
		}

		/**
		 * Display admin notices in case of api credentials not found.
		 */
		public static function wp_travel_display_paystack_notices( $display ) {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			if ( $settings['currency'] != 'NGN' && self::is_enabled() && ! $display ) {
				$display = true;
			}
		}
		/**
		 * Display admin notices in case of api credentials not found.
		 */
		public static function wp_travel_paystack_notices() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			if ( $settings['currency'] != 'NGN' && self::is_enabled() ) {
				$message = sprintf( __( 'Paystack works only with Nigeria Naira.', 'wp-travel-pro' ) );
				printf( '<li ><p>%1$s</p></li>', esc_html( $message ) );
			}

		}
	
		public static function is_disabled() {
			return ! self::is_enabled();
		}

		private static function is_enabled() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			return array_key_exists( 'payment_option_paystack', $settings ) && 'yes' === $settings['payment_option_paystack'];
		}


		public static function uses_paystack_checkout() {
			return isset( $_POST['wp_travel_booking_option'] ) && 'booking_with_payment' === $_POST['wp_travel_booking_option'] && 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'paystack' === $_REQUEST['wp_travel_payment_gateway'];
		}

		public static function booking_message() {
			unset( $_SESSION['used-paystack'] );
			$message = esc_html__( "We've received your booking and payment details. We'll contact you soon.", 'wp-travel-pro' );
			return $message;
		}


		/**
		 * Register Scripts.
		 */
		public static function assets() {
			if ( function_exists( 'wptravel_can_load_payment_scripts' ) ) {
				$load_payment_script = wptravel_can_load_payment_scripts();
			} else {
				$load_payment_script = wp_travel_can_load_payment_scripts();
			}
			if ( $load_payment_script && self::is_enabled() ) {
				$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
				wp_enqueue_script( 'paystack-inline-js', 'https://js.paystack.co/v1/inline.js', array( 'jquery' ), '1.0.0', true );
				wp_enqueue_script( 'paystack-views-js', plugin_dir_url( __FILE__ ) . 'assets/js/dist/paystack-checkout-views' . $suffix . '.js', array( 'paystack-inline-js', 'jquery' ), '1.0.0', true );
			}
		}

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {
			$screen = get_current_screen();
			if (  method_exists( 'WP_Travel',  'is_page' )  ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_PAYSTACK_CHECKOUT_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}


	}
endif;
