<?php
/**
 * PayU Latam Payment Class.
 *
 * @package WP_Travel_PayU_Latam_Checkout_Core
 */

if ( ! class_exists( 'WP_Travel_PayU_Latam_Payment' ) ) :

	/**
	 * PayU Latam Payment Handler.
	 */
	class WP_Travel_PayU_Latam_Payment {

		/**
		 * Defaults Args.
		 *
		 * @var $defaults
		 */
		private $defaults_vars;

		/**
		 * Holds Booking Id.
		 *
		 * @var $booking_id
		 */
		private $booking_id;

		/**
		 * Complete Partial Payment.
		 *
		 * @var $is_complete_partial_payment
		 */
		private $is_complete_partial_payment;

		private $reference_code;

		/**
		 * Constructor.
		 *
		 * @param int  $booking_id Booking ID.
		 * @param bool $complete_partial_payment Is Complete Partial Payment.
		 */
		public function __construct( $booking_id = null, $complete_partial_payment = false ) {
			$this->booking_id                  = $booking_id;
			$this->is_complete_partial_payment = $complete_partial_payment;
			$this->defaults_vars               = array();
			$this->reference_code              = uniqid();
		}

		/**
		 * Get Return URL.
		 */
		public function get_return_url() {
			$trip_id = get_post_meta( (int) $this->booking_id, 'wp_travel_post_id', true );
			$url     = '';
			if ( function_exists( 'wp_travel_thankyou_page_url' ) ) {
				$url = wptravel_thankyou_page_url( (int) $trip_id );
			}
			return $url;
		}

		/**
		 * Merchant's Details Vars.
		 */
		private function get_merchant_details() {
			$args     = array();
			$settings = $this->get_payment_settings();
			extract( $settings ); // $merchant_id, $api_login, $account_id etc
			$return_url = $this->get_return_url();

			if ( function_exists( 'wptravel_test_mode' ) ) {
				$test_mode = wptravel_test_mode();
			} else {
				$test_mode = wp_travel_test_mode();
			}

			// Merchant's Details.
			$args['merchantId']      = $merchant_id;
			$args['api_login']       = $api_login;
			$args['ApiKey']          = $api_key;
			$args['accountId']       = $account_id;
			$args['responseUrl']     = add_query_arg(
				array(
					'booking_id' => $this->booking_id,
					'booked'     => true,
					'status'     => 'success',
					'order_id'   => $this->booking_id,
				),
				$return_url
			);
			$args['tax']             = $tax;
			$args['taxReturnBase']   = $tax_return_base;
			$args['test']            = $test_mode ? 1 : 0;
			$args['lng']             = in_array( get_locale(), array( 'en', 'es', 'pt' ) ) ? get_locale() : 'en';
			$args['referenceCode']   = $this->reference_code;
			$args['confirmationUrl'] = esc_url( add_query_arg( 'payu_latam_listener', 'ITN', home_url( 'index.php' ) ) );
			return $args;
		}

		/**
		 * Buyer's Details Vars.
		 */
		private function get_buyer_details() {
			$booking_data = get_post_meta( (int) $this->booking_id, 'order_data', true );

			$args = array();
			$key  = $this->get_form_index_key();
			if ( $key ) :
				// Buyer Details.
				// $args['name_first']    = isset( $booking_data['wp_travel_fname_traveller'][ $key ][0] ) ? $booking_data['wp_travel_fname_traveller'][ $key ][0] : '';
				// $args['name_last']     = isset( $booking_data['wp_travel_lname_traveller'][ $key ][0] ) ? $booking_data['wp_travel_lname_traveller'][ $key ][0] : '';
				$args['buyerEmail'] = isset( $booking_data['wp_travel_email_traveller'][ $key ][0] ) ? $booking_data['wp_travel_email_traveller'][ $key ][0] : '';
			endif;
			return $args;
		}

		private function get_signature() {
			$settings      = $this->get_payment_settings();
			$arg1          = $this->get_merchant_details();
			$arg2          = $this->get_transaction_details();
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$currency_code = ( isset( $settings['currency'] ) ) ? $settings['currency'] : 'USD';

			$api_key     = $arg1['ApiKey'];
			$merchant_id = $arg1['merchantId'];
			$amount      = $arg2['amount'];
			$currency    = $arg2['currency'];

			$pre_signature_string = "{$api_key}~{$merchant_id}~{$this->reference_code}~{$amount}~{$currency}";

			return array( 'signature' => md5( $pre_signature_string ) );
		}

		/**
		 * Transaction Details.
		 */
		private function get_transaction_details() {
			global $wt_cart;
			$total_price  = $wt_cart->get_total();
			$payment_mode = isset( $_POST['wp_travel_payment_mode'] ) ? $_POST['wp_travel_payment_mode'] : 'full';

			$complete_partial_payment = $this->is_complete_partial_payment;

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			$currency_code = ( isset( $settings['currency'] ) ) ? $settings['currency'] : 'USD';

			$arg     = array();
			$trip_id = get_post_meta( (int) $this->booking_id, 'wp_travel_post_id', true );
			if ( $complete_partial_payment ) {
				$amount        = isset( $_POST['amount'] ) ? $_POST['amount'] : 0;
				$arg['amount'] = $amount;
			} else {
				if ( 'partial' === $payment_mode ) {
					$arg['amount'] = $total_price['total_partial'];
				} else {
					$arg['amount'] = $total_price['total'];
				}
			}
			$arg['description'] = get_the_title( (int) $trip_id );
			$arg['currency']    = strtoupper( $currency_code );

			return $arg;
		}

		public function get_payment_api_url( $ssl_check = false ) {
			return self::get_payment_host();
		}

		public static function get_payment_host( $ssl_check = false ) {
			if ( is_ssl() || ! $ssl_check ) {
				$protocol = 'https://';
			} else {
				$protocol = 'http://';
			}

			if ( function_exists( 'wptravel_test_mode' ) ) {
				$test_mode = wptravel_test_mode();
			} else {
				$test_mode = wp_travel_test_mode();
			}
			if ( $test_mode ) {
				$host = $protocol . 'sandbox.checkout.payulatam.com/ppp-web-gateway-payu/';
			} else {
				$host = $protocol . 'checkout.payulatam.com/ppp-web-gateway-payu/';
			}

			return $host;
		}

		/**
		 * Checkout Form Index Key.
		 */
		private function get_form_index_key() {
			$order_details = get_post_meta( (int) $this->booking_id, 'order_items_data', true ); // Multiple Trips.

			$index = is_array( $order_details ) && count( $order_details ) > 0 ? array_keys( $order_details )[0] : null;
			return $index;
		}

		/**
		 * Get all the payment fields values.
		 *
		 * @return array
		 */
		public function get_payment_args() {
			$arg1 = $this->get_merchant_details();
			$arg2 = $this->get_buyer_details();
			$arg3 = $this->get_transaction_details();
			$arg4 = $this->get_custom_args();
			$arg5 = $this->get_signature();
			return array_merge( $arg1, $arg2, $arg3, $arg4, $arg5 );
		}

		/**
		 * Extra custom form variables.
		 *
		 * @return []
		 */
		public function get_custom_args() {
			$arg = array();

			$arg['extra2'] = (int) $this->booking_id;
			return $arg;
		}

		/**
		 * Prepare final payment form to be submitted.
		 *
		 * @return string
		 */
		public function get_payment_form() {
			$params   = array(
				'amount',
				'merchantId',
				'referenceCode',
				'accountId',
				'description',
				'tax',
				'taxReturnBase',
				'signature',
				'currency',
				'buyerEmail',
				'extra2',
				'responseUrl',
				'confirmationUrl',
			);
			$args     = $this->get_payment_args();
			$url_keys = array( 'responseUrl', 'confirmationUrl' );
			$api_url  = $this->get_payment_api_url();
			?>
			<form id="wp_travel_payu_latam_payment_form" method="post" action="<?php echo esc_url( $api_url ); ?>">
				<h3><?php esc_html_e( 'You will redirect shortly ...', 'wp-travel-pro' ); ?></h3>
				<?php
				foreach ( $params as $key ) :
					if ( ! isset( $args[ $key ] ) ) {
						continue;
					}

					if ( array_key_exists( $key, $url_keys ) ) {
						echo '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_url( $args[ $key ] ) . '">';
						continue;
					}
					echo '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_attr( $args[ $key ] ) . '">';
				endforeach;
				?>
				<input type="submit" value="Pay"/>
			</form>
			<script>
				var form = document.getElementById( 'wp_travel_payu_latam_payment_form' );
				form.querySelector('input[type=submit]').style.display = 'none';
				form.submit();
			</script>
			<?php
		}

		/**
		 * Payment Settings.
		 */
		private function get_payment_settings() {
			$settings_instance = WP_Travel_PayU_Latam_Settings::instance();
			$settings          = $settings_instance->get_settings();
			return $settings;
		}
	}

endif;
