<?php
/**
 * Modules core file.
 *
 * @package wp-travel-payu-latam-checkout-core
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

// WP Travel PayU Latam Checkout Core.
if ( ! class_exists( 'WP_Travel_PayU_Latam_Checkout_Core' ) ) :
	/**
	 * WP Travel PayU Latam Checkout Core.
	 */
	class WP_Travel_PayU_Latam_Checkout_Core {

		const WP_TRAVEL_PAYU_LATAM_CHECKOUT_HANDLE = 'wp_travel_payu_latam_checkout_';

		/**
		 * Absolute path to core
		 *
		 * @var string
		 */
		protected static $abspath;

		/**
		 * Plugin File.
		 *
		 * @var [type]
		 */
		protected static $plugin_file;

		/**
		 * Plugin Version.
		 *
		 * @var string
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP_Travel_PayU_Latam_Checkout_Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_PayU_Latam_Checkout_Core Instance.
		 * Ensures only one instance of WP_Travel_PayU_Latam_Checkout_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_PayU_Latam_Checkout_Core()
		 * @return WP_Travel_PayU_Latam_Checkout_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param Array $params Core class init paramerters.
		 */
		public static function init( $params ) {
			self::$abspath     = $params['abspath'] . 'inc/modules/payu-latam-checkout-core/';
			self::$plugin_file = __FILE__;
			self::$version     = $params['version'];
			self::includes();
			self::init_hooks();
		}

		/**
		 * Includes required files.
		 */
		public static function includes() {
			include_once self::$abspath . 'inc/admin/settings.php';
			include_once self::$abspath . 'inc/functions.php';
			include_once self::$abspath . 'inc/lib/PayU.php';
		}

		/**
		 * Init Hooks.
		 */
		public static function init_hooks() {
			add_action( 'wp_enqueue_scripts', array( __CLASS__, 'assets' ) );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'admin_assets' ) );
			if ( self::uses_payu_latam_checkout() ) {
				add_action( 'wp_travel_after_frontend_booking_save', array( __CLASS__, 'process' ), 20, 1 );
				add_action( 'wp_travel_before_partial_payment_complete', array( __CLASS__, 'process' ), 10, 2 );
			}

			// General Notices.
			// add_filter( 'wp_travel_display_general_admin_notices', array( __CLASS__, 'display_notices' ), 20 );
			// add_action( 'wp_travel_general_admin_notice', array( __CLASS__, 'notices' ), 20 );

			if ( isset( $_SESSION['used-payu-latam'] ) && $_SESSION['used-payu-latam'] ) {
				add_filter( 'wp_travel_booked_message', array( __CLASS__, 'booking_message' ), 20 );
			}

		}

		/**
		 * Process After Checkout Form Submit.
		 *
		 * @param int     $booking_id Booking ID.
		 * @param boolean $complete_partial_payment Complete Payment Process.
		 * @return void
		 */
		public static function process( $booking_id, $complete_partial_payment = false ) {
			if ( ! $booking_id || self::is_disabled() || ! self::uses_payu_latam_checkout() ) {
				return;
			}

			if ( ! $complete_partial_payment ) {
				do_action( 'wt_before_payment_process', $booking_id );
			}

			self::web_redirect( $booking_id, $complete_partial_payment );
			exit;
		}

		/**
		 * Redirect Using Script.
		 *
		 * @param int $booking_id Booking ID.
		 * @param bool $complete_partial_payment Complete Partial Payment.
		 * @since 1.0.0
		 */
		public static function web_redirect( $booking_id, $complete_partial_payment ) {

			include_once plugin_dir_path( __FILE__ ) . 'inc/class-wp-travel-payu-latam-payment.php';

			header( 'Content-type: text/html; charset=utf-8' );
			$pf   = new WP_Travel_PayU_Latam_Payment( $booking_id, $complete_partial_payment );
			$form = $pf->get_payment_form();
			?>
			<!DOCTYPE html>
			<html lang="en">
			<head>
				<meta charset="UTF-8">
				<meta name="viewport" content="width=device-width, initial-scale=1.0">
				<meta http-equiv="X-UA-Compatible" content="ie=edge">
				<title>PayU Latam Payment</title>
			</head>
			<body>
				<?php echo wp_kses_post( $form ); ?>
			</body>
			</html>
			<?php
			exit;
		}

		/**
		 * Display admin notices in case of API credentials not found.
		 *
		 * @param bool $display Display Notice.
		 */
		public static function display_notices( $display ) {
			return false;
		}

		/**
		 * Adds Notice if currency not matched.
		 */
		public static function notices() {}

		/**
		 * Chekcs if this payment is disabled.
		 *
		 * @return boolean
		 */
		public static function is_disabled() {
			return ! self::is_enabled();
		}

		/**
		 * Checks if this payent is enabled.
		 *
		 * @return boolean
		 */
		private static function is_enabled() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			return isset( $settings['wp_travel_payu_latam_settings'] ) && array_key_exists( 'payment_option_payu_latam', $settings['wp_travel_payu_latam_settings'] ) && 'yes' === $settings['wp_travel_payu_latam_settings']['payment_option_payu_latam'];
		}


		/**
		 * If current booking uses PayU Latam.
		 *
		 * @return boolean
		 */
		public static function uses_payu_latam_checkout() {
			$settings       = wptravel_get_settings();
			if( isset( $settings['selected_booking_option'] ) &&  count( $settings['selected_booking_option'] ) == 1 && $settings['selected_booking_option'][0] = 'booking-with-payment' ){ 
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'payu_latam' === $_REQUEST['wp_travel_payment_gateway'];
			}else{
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_booking_option', $_REQUEST ) && 'booking_with_payment' === $_REQUEST['wp_travel_booking_option'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'payu_latam' === $_REQUEST['wp_travel_payment_gateway'];
			}

		}

		/**
		 * Booking Message After booking.
		 *
		 * @return string
		 */
		public static function booking_message() {
			unset( $_SESSION['used-payu-latam'] );
			$message = esc_html__( "We've received your booking and payment details. We'll contact you soon.", 'wp-travel-pro' );
			return $message;
		}


		/**
		 * Register/Enqueue Scripts.
		 */
		public static function assets() {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			// wp_enqueue_script( 'payu-latam-view-js', plugin_dir_url( __FILE__ ) . 'assets/js/wp-travel-payu-latam-checkout-view' . $suffix . '.js', array( 'jquery' ), '1.0.0', true );
		}

		/**
		 * Admin scripts.
		 */
		public static function admin_assets() {
			$screen = get_current_screen();
			// settings_screen.
			if ( WP_Travel::is_page( 'settings', true ) ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_PAYU_LATAM_CHECKOUT_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}


	}
endif;


