(function () {
    function getStorageId(form) {
        return 'formdata_' + location.pathname + '_' + (form.id || '');
    }

    function extractValues(form) {
        const formData = new FormData(form);
        let preparedData = {};

        formData.forEach((value, key) => {
            if (preparedData[key]) {
                preparedData[key] = Array.isArray(preparedData[key])
                    ? preparedData[key].concat(encodeURIComponent(value))
                    : [preparedData[key], encodeURIComponent(value)];
            } else {
                preparedData[key] = encodeURIComponent(value);
            }
        });

        return preparedData;
    }

    function fillFields(form, formData) {
        Object.keys(formData).forEach((name) => {
            let field = form.elements[name];
            if (!field) return; // ✅ Prevents the error by skipping undefined elements
    
            let tagName = field.tagName?.toLowerCase(); // ✅ Safe access to `tagName`
            let inputType = field.type;
    
            let value = formData[name];
            value = Array.isArray(value) ? value.map(decodeURIComponent) : decodeURIComponent(value);
    
            if (inputType === 'checkbox') {
                if (Array.isArray(value)) {
                    value.forEach((val) => {
                        let checkbox = form.querySelector(`[name="${name}"][value="${val}"]`);
                        if (checkbox) checkbox.checked = true;
                    });
                } else {
                    let checkbox = form.querySelector(`[name="${name}"][value="${value}"]`);
                    if (checkbox) checkbox.checked = true;
                }
            } else if (inputType === 'radio') {
                let radio = form.querySelector(`[name="${name}"][value="${value}"]`);
                if (radio) radio.checked = true;
            } else {
                field.value = value;
            }
        });
    }
    

    function saveForm(form) {
        const storageId = getStorageId(form);
        localStorage.setItem(storageId, JSON.stringify(extractValues(form)));
    }

    function restoreForm(form) {
        const storageId = getStorageId(form);
        const savedData = localStorage.getItem(storageId);
        if (savedData) {
            fillFields(form, JSON.parse(savedData));
            localStorage.removeItem(storageId); // Clears the stored data after restoring
        }
    }

    // Attach save functionality on form input changes
    document.addEventListener("DOMContentLoaded", function () {
        document.querySelectorAll("form").forEach((form) => {
            form.addEventListener("input", () => saveForm(form));
            restoreForm(form); // Restore data on page load
        });
    });
})();
