<?php
/**
 * WP Travel Pro Core Class.
 *
 * @package wp-travel-pro-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Pro core.
if ( ! class_exists( 'WP_Travel_Pro_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Pro_Core {
		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Pro Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Pro_Core Instance.
		 * Ensures only one instance of WP_Travel_Pro_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Pro_Core()
		 * @return WP_Travel_Pro_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			self::includes();

			// add_action( 'admin_init', array( 'WP_Travel_Pro_Core', 'uninstall_addons' ) );
			add_filter( 'wp_travel_show_upsell_message', array( 'WP_Travel_Pro_Core', 'upsell_display' ), 10, 2 );

			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Pro_Core', 'admin_assets' ), 20 );
			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Pro_Core', 'frontend_assets' ), 15 );

			add_filter(
				'wp-travel-pro-bundle-items',
				function( $handles ) {
					$handles['scripts'] = array(
						'wp-travel-advanced-gallery-view',
						'wp-travel-tour-extras-front-end-js',
						'wp-travel-tour-extras-cart-js',
						'wp_travel_trip_weather_forecast_frontend_scripts',
						'wp-travel-utilities-front-end-js',
						'wp-travel-list-wishlists',
						'wp-travel-wishlists',
						'wp_travel_multiple_currency_frontend_scripts',
					);

					$handles['styles'] = array(
						'wp-travel-utilities-front-end-css',
						'wp-travel-list-wishlists',
						'wp-travel-advanced-gallery-view',
						'wp_travel_downloads_frontend_css',
						'wp-travel-tour-extras-front-end-css',
						'wp_travel_trip_weather_forecast_frontend_styles',
					);

					return $handles;
				}
			);

			// Add cutoff time in get trip data.
			add_filter( 'wp_travel_trip_data', array( 'WP_Travel_Pro_Core', 'additional_trip_data' ), 10, 2 );
			// update cutoff time in get trip data.
			add_action( 'wp_travel_update_trip_data', array( 'WP_Travel_Pro_Core', 'update_additional_trip_data' ), 10, 2 );
			add_filter( 'slicewp_is_integration_plugin_active_wp_travel', function ( $arg ) { return true; } );
			add_filter( 'wp_travel_all_booking_data_list_for_slicewp', array( 'WP_Travel_Pro_Core', 'wp_travel_conpatible_with_slicewp' ), 10, 2 );
			add_filter( 'wp_travel_booking_status_trigger_from', array( 'WP_Travel_Pro_Core', 'wp_travel_slicewp_commission_status_update' ), 10, 2 );
			add_action( 'trashed_post', array( 'WP_Travel_Pro_Core', 'wp_travel_booking_move_trash' ) );
			add_action( 'untrash_post', array( 'WP_Travel_Pro_Core', 'wp_travel_post_type_status_change' ), 10, 2 );
			add_action( 'publish_to_draft', array( 'WP_Travel_Pro_Core', 'wp_travel_trigger_post_type_status_change_publish_to_draft' ) );
			add_action( 'draft_to_publish', array( 'WP_Travel_Pro_Core', 'wp_travel_trigger_post_type_status_change_draft_to_publish' ) );
		}

		public static function additional_trip_data( $trip_data, $trip_id ) {
			$trip_data['cuttOffTime'] = get_post_meta( $trip_id, 'cuttOffTime', true );
			return $trip_data;
		}
		public static function update_additional_trip_data( $trip_data, $trip_id ) {
			if ( isset( $trip_data->cuttOffTime ) ) {
				update_post_meta( $trip_id, 'cuttOffTime', wp_unslash( $trip_data->cuttOffTime ) );
			}
		}

		public static function upsell_display( $display, $type ) {
			return false;
		}

		/**
		 * Frontend assets.
		 */
		public static function frontend_assets() {

			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			wp_register_script( 'wp-travel-pro-bundle', self::$plugin_url . 'assets/js/wp-travel-pro-bundle.js', array( 'wp-travel-frontend-bundle', 'wp-util' ), self::$version, true );
			wp_register_style( 'wp-travel-pro-bundle', self::$plugin_url . 'assets/css/wp-travel-pro-bundle.css', array(), self::$version );
			if ( is_singular( WP_TRAVEL_POST_TYPE ) ) {
				wp_enqueue_script( 'wp-travel-pro-frontend', self::$plugin_url . 'app/build/frontend-booking-widget.js', array( 'wp-data', 'wp-element', 'wp-hooks', 'wp-polyfill' ), self::$version, true );
			}

			wp_register_script( 'wp-travel-formsaver-script', self::$plugin_url . 'assets/js/formsaver.js', array(), self::$version, true );
			wp_enqueue_script( 'wp-travel-formsaver-script' );
		}	

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {

			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			// temporary fix for show hide payment addons fields on payment settings tab.
			wp_enqueue_script( 'wp-travel-pro-script', self::$plugin_url . 'assets/js/wp-travel-pro.js', array( 'jquery' ), WP_TRAVEL_VERSION, true );

			$deps                   = include_once sprintf( '%sapp/build/admin-trip-options.asset.php', self::$abspath );
			$deps['dependencies'][] = 'wp-travel-admin-trip-options';
			$deps['dependencies'][] = 'wp-travel-admin-script';
			wp_enqueue_script( 'wp-travel-pro-admin-trip-option', self::$plugin_url . 'app/build/admin-trip-options.js', $deps['dependencies'], $deps['version'], true );
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}

			if( apply_filters( 'wptravel_enable_pwa_service', false ) == true ){
				require_once 'core/class-pwa.php';
			}

			require_once 'core/trip-quick-view.php';
			
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}
		/**
		 * use to compatible slicewp by using hook
		 * @param int $booking_id
		 * @param int $user_id
		 */
		public static function wp_travel_conpatible_with_slicewp( $booking_id, $user_id ) {
			if ( function_exists(  'slicewp_get_setting' ) ) {
				$slicewp_settings = slicewp_get_setting( 'active_integrations' );
				if ( ! empty( $slicewp_settings ) && is_array( $slicewp_settings ) ) {
					if ( ! in_array( 'wptravel', $slicewp_settings ) )
						return;
				} else {
					return;
				}
			} else {
				return;
			}
			$payment_data = wptravel_booking_data( $booking_id );
			$traveler_info = get_post_meta( $booking_id, 'order_data', true );
			$traveler_info = maybe_unserialize( $traveler_info );
			$commission_data = apply_filters( 'wp_travel_slicewp_affiliate_commission_id', array(), $booking_id, $user_id, $traveler_info, $payment_data );
			if ( ! empty( $commission_data ) ) {
				update_post_meta( $booking_id, 'slicewp_commission_data', $commission_data );
			}
		}
		/**
		 * use to update commission status while update booking status
		 * @param int booking_id
		 * @param string booking_status
		 */
		public static function wp_travel_slicewp_commission_status_update( $booking_id, $booking_status ) {
			if ( function_exists(  'slicewp_get_setting' ) ) {
				$slicewp_settings = slicewp_get_setting( 'active_integrations' );
				if ( ! empty( $slicewp_settings ) && is_array( $slicewp_settings ) ) {
					if ( ! in_array( 'wptravel', $slicewp_settings ) )
						return;
				} else {
					return;
				}
			} else {
				return;
			}
			$payment_data = wptravel_booking_data( $booking_id );
			$payment_mode = isset( $payment_data['payment_mode'] ) ? $payment_data['payment_mode'] : 'N/A';
			$payment_status = isset( $payment_data['payment_status'] ) ? $payment_data['payment_status'] : 'pending';
			$paid_amount = isset( $payment_data['paid_amount'] ) ? $payment_data['paid_amount'] : 0;
			$total = isset( $payment_data['total'] ) ? $payment_data['total'] : 0;
			if ( $payment_mode == 'Full' && $total > 0 && $payment_status == 'paid' && $total == $paid_amount ) {
				$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'pending', $booking_id, 'unpaid' );
			} else  {
				$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'pending', $booking_id, 'pending' );
			}

		}
		/**
		 * trigger while wp travel booking move publish to trash
		 * @param int $booking_id
		 */
		public static function wp_travel_booking_move_trash( $booking_id ) {
			if ( function_exists(  'slicewp_get_setting' ) ) {
				$slicewp_settings = slicewp_get_setting( 'active_integrations' );
				if ( ! empty( $slicewp_settings ) && is_array( $slicewp_settings ) ) {
					if ( ! in_array( 'wptravel', $slicewp_settings ) )
						return;
				} else {
					return;
				}
			} else {
				return;
			}

			if ( get_post_type( $booking_id ) != 'itinerary-booking' )
				return;
			$affiliate_data = apply_filters( 'wp_travel_booking_status_trigger', 'reject', $booking_id, 'rejected' );
		}
		/**
		 * trigger while any change of status in trash folder of wp travel booking
		 * @param int $booking_id
		 * @param string previous status
		 * 
		 */
		public static function wp_travel_post_type_status_change( $booking_id, $pre_status ) {
			if ( function_exists(  'slicewp_get_setting' ) ) {
				$slicewp_settings = slicewp_get_setting( 'active_integrations' );
				if ( ! empty( $slicewp_settings ) && is_array( $slicewp_settings ) ) {
					if ( ! in_array( 'wptravel', $slicewp_settings ) )
						return;
				} else {
					return;
				}
			} else {
				return;
			}
			$payment_data = wptravel_booking_data( $booking_id );
			$payment_mode = isset( $payment_data['payment_mode'] ) ? $payment_data['payment_mode'] : 'N/A';
			$payment_status = isset( $payment_data['payment_status'] ) ? $payment_data['payment_status'] : 'pending';
			$paid_amount = isset( $payment_data['paid_amount'] ) ? $payment_data['paid_amount'] : 0;
			$total = isset( $payment_data['total'] ) ? $payment_data['total'] : 0;
			if ( $pre_status == 'publish' ) {
				if ( $payment_mode == 'Full' && $total > 0 && $payment_status == 'paid' && $total == $paid_amount ) {
					$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'unpaid', $booking_id, 'unpaid' );
				} else  {
					$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'pending', $booking_id, 'pending' );
				}
			} elseif ( $pre_status == 'draft' ) {
				$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'rejected', $booking_id, 'rejected' );
			} else {
				$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'pending', $booking_id, 'pending' );
			}
		}
		/**
		 * trigger when any status change publish_to_draft
		 * 
		 * @param class WP_Post
		 */
		public static function wp_travel_trigger_post_type_status_change_publish_to_draft( $booking_post ) {
			if ( function_exists(  'slicewp_get_setting' ) ) {
				$slicewp_settings = slicewp_get_setting( 'active_integrations' );
				if ( ! empty( $slicewp_settings ) && is_array( $slicewp_settings ) ) {
					if ( ! in_array( 'wptravel', $slicewp_settings ) )
						return;
				} else {
					return;
				}
			} else {
				return;
			}
			$booking_id = 0;
			if ( is_a( $booking_post, 'WP_Post' ) )
				$booking_id = $booking_post->ID;
			
			if ( ! empty( $booking_id ) && $booking_id < 1 ) 
				return;
			
			if ( empty( $booking_id ) )
				return;
			
			if ( get_post_type( $booking_id ) != 'itinerary-booking' )
				return;
			
			$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'rejected', $booking_id, 'rejected' );
			return;
		}
		/**
		 * trigger when any status change draft_to_publish
		 * 
		 * @param class WP_Post
		 */
		public static function wp_travel_trigger_post_type_status_change_draft_to_publish( $booking_post ) {
			if ( function_exists(  'slicewp_get_setting' ) ) {
				$slicewp_settings = slicewp_get_setting( 'active_integrations' );
				if ( ! empty( $slicewp_settings ) && is_array( $slicewp_settings ) ) {
					if ( ! in_array( 'wptravel', $slicewp_settings ) )
						return;
				} else {
					return;
				}
			} else {
				return;
			}
			$booking_id = 0;
			if ( is_a( $booking_post, 'WP_Post' ) )
				$booking_id = $booking_post->ID;
			
			if ( ! empty( $booking_id ) && $booking_id < 1 ) 
				return;
			
			if ( empty( $booking_id ) )
				return;
			
			if ( get_post_type( $booking_id ) != 'itinerary-booking' )
				return;
				
			$payment_data = wptravel_booking_data( $booking_id );
			$payment_mode = isset( $payment_data['payment_mode'] ) ? $payment_data['payment_mode'] : 'N/A';
			$payment_status = isset( $payment_data['payment_status'] ) ? $payment_data['payment_status'] : 'pending';
			$paid_amount = isset( $payment_data['paid_amount'] ) ? $payment_data['paid_amount'] : 0;
			$total = isset( $payment_data['total'] ) ? $payment_data['total'] : 0;
			if ( $payment_mode == 'Full' && $total > 0 && $payment_status == 'paid' && $total == $paid_amount ) {
				$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'pending', $booking_id, 'unpaid' );
			} else  {
				$affiliate_status = apply_filters( 'wp_travel_booking_status_trigger', 'pending', $booking_id, 'pending' );
			}
		}
	}
endif;
