<?php
/**
 * WP Travel Razorpay Checkout Core Class.
 *
 * @package wp-travel-razorpay-checkout-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
// WP Travel Razorpay Checkout core.
if ( ! class_exists( 'WP_Travel_Razorpay_Checkout_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Razorpay_Checkout_Core {

		const WP_TRAVEL_RAZORPAY_CHECKOUT_HANDLE = 'wp_travel_razorpay_checkout_';
		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Razorpay Checkout Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Sandbox Url
		 *
		 * @var string
		 */
		public static $sandbox_url = 'https://test.razorpay.com/api/1.1/payment-requests/';

		/**
		 * Production Url
		 *
		 * @var string
		 */
		public static $production_url = 'https://razorpay.com/api/1.1/payment-requests/';

		/**
		 * Main WP_Travel_Razorpay_Checkout_Core Instance.
		 * Ensures only one instance of WP_Travel_Razorpay_Checkout_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Razorpay_Checkout_Core()
		 * @return WP_Travel_Razorpay_Checkout_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			self::includes();

			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Razorpay_Checkout_Core', 'frontend_assets' ) );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Razorpay_Checkout_Core', 'admin_assets' ) );

			// Payment Gateway list.
			add_filter( 'wp_travel_payment_gateway_lists', 'wp_travel_gateway_razprpay' );
			add_filter( 'wp_travel_premium_addons_list', 'wp_travel_razorpay_addons' );

			if ( self::is_enabled() ) {
				add_filter( 'wptravel_frontend_data', 'wp_travel_razorpay_add_vars', 10, 2 );
			}

			if ( self::uses_razorpay_checkout() ) {
				// This will call after payement completed.
				add_action( 'wp_travel_after_frontend_booking_save', array( 'WP_Travel_Razorpay_Checkout_Core', 'set_up_payment_options' ) );
				add_action( 'wp_travel_after_partial_payment_complete', array( 'WP_Travel_Razorpay_Checkout_Core', 'set_up_partial_payment' ) );
			}
			// Razorpay Ajax req.
			if ( isset( $_GET['razorpay_amount'] ) ) {
				add_action( 'init', array( 'WP_Travel_Razorpay_Checkout_Core', 'wp_travel_razorpay_ajax_request' ) );
			}

			isset( $_SESSION['used-razorpay'] ) && $_SESSION['used-razorpay'] && add_filter( 'wp_travel_booked_message', array( 'WP_Travel_Razorpay_Checkout_Core', 'booking_message' ) );

			// General Notices.
			add_filter( 'wp_travel_display_general_admin_notices', array( 'WP_Travel_Razorpay_Checkout_Core', 'wp_travel_display_razorpay_notices' ), 20 );
			add_action( 'wp_travel_general_admin_notice', array( 'WP_Travel_Razorpay_Checkout_Core', 'wp_travel_razorpay_notices' ), 20 );

			add_action('wp_ajax_create_razorpay_order', array( 'WP_Travel_Razorpay_Checkout_Core', 'create_razorpay_order' ));
			add_action('wp_ajax_nopriv_create_razorpay_order', array( 'WP_Travel_Razorpay_Checkout_Core', 'create_razorpay_order' )); // For non-logged-in users

		}

		public static function create_razorpay_order() {
			
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			// Get data sent via AJAX
			$amount = isset($_POST['amount']) ? intval($_POST['amount']) : 0;
			$currency = isset($_POST['currency']) ? sanitize_text_field($_POST['currency']) : 'INR';
			$name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
			$email = isset($_POST['email']) ? sanitize_text_field($_POST['email']) : '';
			$contact = isset($_POST['contact']) ? sanitize_text_field($_POST['contact']) : '';
			$address = isset($_POST['address']) ? sanitize_text_field($_POST['address']) : '';
		
			if ($amount <= 0) {
				wp_send_json_error(['message' => 'Invalid amount']);
			}
		
			// Set your Razorpay API key and secret
			$razorpay_key_id = $settings['wp_travel_razorpay_checkout']['api_key']; // Replace with your Razorpay key ID
			$razorpay_key_secret = $settings['wp_travel_razorpay_checkout']['auth_token']; // Replace with your Razorpay key secret
		
			// Prepare the request data
			$order_data = [
				'amount'         => $amount,
				'currency'       => $currency
			];
		
			// Set the headers for authorization
			$auth = base64_encode($razorpay_key_id . ':' . $razorpay_key_secret);
		
			// Make the POST request to Razorpay to create the order
			$response = wp_remote_post('https://api.razorpay.com/v1/orders', [
				'method'    => 'POST',
				'headers'   => [
					'Content-Type'  => 'application/json',
					'Authorization' => 'Basic ' . $auth
				],
				'body'      => json_encode($order_data),
			]);

			// Check for errors in the request
			if (is_wp_error($response)) {
				wp_send_json_error(['message' => 'Error creating Razorpay order', 'error' => $response->get_error_message()]);
			}
		
			// Parse the response
			$order_data = json_decode(wp_remote_retrieve_body($response), true);
		
			// Check if the order ID is returned
			if (isset($order_data)) {
				wp_send_json_success(['order_id' => $order_data]); // Return the order ID
			} else {
				wp_send_json_error(['message' => 'Failed to create Razorpay order']);
			}
		}

		/**
		 * Determine if booking used express checkout.
		 */
		private static function uses_razorpay_checkout() {
			return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_booking_option', $_REQUEST ) && 'booking_with_payment' === $_REQUEST['wp_travel_booking_option'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'razorpay_checkout' === $_REQUEST['wp_travel_payment_gateway'];
		}

		/**
		 * Alter the booking travel message as payment has already been done.
		 *
		 * @param string $message Set trip booked message.
		 * @return string
		 */
		public static function booking_message( $message ) {

			unset( $_SESSION['used-razorpay'] );

			$message = esc_html__( "We've received your booking and payment details. We'll contact you soon.", 'wp-travel-pro' );

			return $message;

		}

		/**
		 * Determine if Razorpay checkout is enabled.
		 */
		private static function is_enabled() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			return array_key_exists( 'payment_option_razorpay_checkout', $settings ) && 'yes' === $settings['payment_option_razorpay_checkout'];
		}

		/**
		 * Determing is Razorpay checkout is disabled.
		 */
		private static function is_disabled() {
			return ! self::is_enabled();
		}

		/**
		 * Razorpay Frontend assets.
		 */
		public static function frontend_assets() {

			if( isset( wptravel_get_settings()["enable_one_page_booking"] ) && wptravel_get_settings()["enable_one_page_booking"] == "1" ){
				wp_enqueue_script( 'wp-travel-razorpay-checkout-server', 'https://checkout.razorpay.com/v1/checkout.js', array(), '0.0.1', true );
				wp_enqueue_script( 'wp-travel-razorpay-onpage-checkout', self::$plugin_url . 'assets/js/wp-travel-razorpay-onpage-checkout.js', array( 'wp-travel-razorpay-checkout-server' ), self::$version, true );
				// wp_localize_script( 'wp-travel-razorpay-onpage-checkout', 'wp_travel_onpage_booking', wptravel_get_settings() );
				wp_localize_script( 'wp-travel-razorpay-onpage-checkout', 
					'wp_travel_onpage_booking', 
					array( 
						'currency' =>  wptravel_get_settings()['currency'],
						'one_page_booking' => wptravel_get_settings()["enable_one_page_booking"] 
					) 
				);
			}
			if ( function_exists( 'wptravel_can_load_payment_scripts' ) ) {
				$load_payment_script = wptravel_can_load_payment_scripts();
			} else {
				$load_payment_script = wp_travel_can_load_payment_scripts();
			}
			if ( $load_payment_script && self::is_enabled() && wptravel_get_settings()["enable_one_page_booking"] !== "1" ) {

				$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
				wp_enqueue_script( 'wp-travel-razorpay-checkout-server', 'https://checkout.razorpay.com/v1/checkout.js', array(), '0.0.1', true );
				wp_enqueue_script( 'wp-travel-razorpay-checkout', self::$plugin_url . 'assets/js/wp-travel-razorpay-checkout' . $suffix . '.js', array( 'wp-travel-razorpay-checkout-server' ), self::$version, true );

			}
		}

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {

			$suffix         = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$screen         = get_current_screen();
			$allowed_screen = array( 'edit-itineraries', WP_TRAVEL_POST_TYPE, 'itineraries_page_settings', 'itinerary-booking', 'itinerary-booking_page_settings' );

			$screen = get_current_screen();
			if (  method_exists( 'WP_Travel',  'is_page' )  ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_RAZORPAY_CHECKOUT_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		/**
		 * Display admin notices in case of api credentials not found.
		 */
		public static function wp_travel_display_razorpay_notices( $display ) {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			if ( $settings['currency'] != 'INR' && self::is_enabled() && ! $display ) {
				$display = true;
			}
		}
		/**
		 * Display admin notices in case of api credentials not found.
		 */
		public static function wp_travel_razorpay_notices() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			if ( $settings['currency'] != 'INR' && self::is_enabled() ) {
				$message = sprintf( __( 'Razorpay works only with Indian Currency.', 'wp-travel-pro' ) );
				printf( '<li ><p>%1$s</p></li>', $message );
			}

		}

		public static function wp_travel_razorpay_ajax_request() {
			try {
				error_reporting( 0 );

				header( 'Content-Type: application/json' );

				if ( function_exists( 'wptravel_get_settings' ) ) {
					$settings = wptravel_get_settings();
				} else {
					$settings = wp_travel_get_settings();
				}

				if ( ! isset( $settings['currency'] ) || ! ( isset( $settings['currency'] ) && $settings['currency'] == 'INR' ) ) {
					die(
						json_encode(
							[
								'message' => 'Razorpay only supports indian currency (INR)',
								'success' => false,
							]
						)
					);
				}

				if ( ! ( $settings['wp_travel_razorpay_checkout']['api_key'] && $settings['wp_travel_razorpay_checkout']['auth_token'] ) ) {
					die(
						json_encode(
							[
								'message' => 'API KEYS ARE NOT SET.',
								'success' => false,
							]
						)
					);
				}

				ob_start();

				try {
					$ch = curl_init();

					if ( function_exists( 'wptravel_test_mode' ) ) {
						$test_mode = wptravel_test_mode();
					} else {
						$test_mode = wp_travel_test_mode();
					}
					curl_setopt( $ch, CURLOPT_URL, $test_mode ? $this->sandbox_url : $this->production_url );
					curl_setopt( $ch, CURLOPT_HEADER, false );
					curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
					curl_setopt( $ch, CURLOPT_FOLLOWLOCATION, true );
					curl_setopt(
						$ch,
						CURLOPT_HTTPHEADER,
						array(
							'X-Api-Key:' . $settings['wp_travel_razorpay_checkout']['api_key'],
							'X-Auth-Token:' . $settings['wp_travel_razorpay_checkout']['auth_token'],
						)
					);

					$payload = array(
						'purpose'                 => $_GET['for'],
						'amount'                  => $_GET['razorpay_amount'],
						'buyer_name'              => implode( ' ', [ $_GET['first_name'], $_GET['last_name'] ] ),
						'send_email'              => false,
						'send_sms'                => false,
						'email'                   => $_GET['email'],
						'allow_repeated_payments' => false,
					);

					if ( isset( $_GET['phone'] ) && strlen( ( $_GET['phone'] ) ) == 10 ) {
						$payload['phone'] = $_GET['phone'];
					}
					curl_setopt( $ch, CURLOPT_POST, true );
					curl_setopt( $ch, CURLOPT_POSTFIELDS, http_build_query( $payload ) );
					$response = curl_exec( $ch );

					curl_close( $ch );
					echo $response;
				} catch ( \Exception $e ) {

				}
			} catch ( \Exception $e ) {

			}
			exit();
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			include sprintf( '%sinc/helpers.php', self::$abspath );
			include sprintf( '%sinc/functions.php', self::$abspath );
			if ( self::is_request( 'admin' ) ) {
				include sprintf( '%sinc/admin/settings.php', self::$abspath );
			}
		}

		/**
		 * Sets up payment options
		 *
		 * @param string $booking_id ID of booking.
		 * @return void
		 */
		public static function set_up_payment_options( $booking_id ) {

			if ( self::is_disabled() ) {
				return;
			}

			if ( ! self::uses_razorpay_checkout() ) {
				return;
			}

			if ( ! $booking_id ) {
				return;
			}
			if ( ! isset( $_POST['wp_travel_book_now'] ) ) {
				return;
			}

			do_action( 'wt_before_payment_process', $booking_id );

			$json = sanitize_text_field( wp_unslash( $_REQUEST['payment_details'] ) );

			$detail = json_decode( $json );
			$amount = isset( $detail->amount ) ? $detail->amount : 0;

			$payment_id     = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
			$payment_method = 'razorpay_checkout';
			update_post_meta( $payment_id, 'wp_travel_payment_gateway', $payment_method );

			wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_method ), $payment_id );

			$_SESSION['used-razorpay'] = true;

			do_action( 'wp_travel_after_successful_payment', $booking_id );
		}

		/**
		 * Sets up payment options
		 *
		 * @return void
		 */
		public static function set_up_partial_payment() {

			if ( self::is_disabled() ) {
				return;
			}

			if ( ! self::uses_razorpay_checkout() ) {
				return;
			}

			if ( ! isset( $_POST['complete_partial_payment'] ) ) {
				return;
			}

			if ( ! isset( $_POST['wp_travel_booking_id'] ) ) {
				return;
			}
			$payment_gateway = sanitize_text_field( wp_unslash( $_POST['wp_travel_payment_gateway'] ) );
			$booking_id      = sanitize_text_field( wp_unslash( $_POST['wp_travel_booking_id'] ) );
			// get previous payment ids.
			$payment_id      = get_post_meta( $booking_id, 'wp_travel_payment_id', true );

			/**
			 * This will return the current payment id to insert payment meta for the current payment.
			 * 
			 * @hooked wptravel_update_payment_post from partial payment which will also update booking option [booking_only | booking_with_payment ]
			 *
			 * @since 4.0.2
			 */
			$new_payment_id = apply_filters( 'wptravel_before_insert_partial_payment', $payment_id, $booking_id, $payment_gateway );

			$json = sanitize_text_field( wp_unslash( $_POST['payment_details'] ) );

			$detail = json_decode( $json );
			$amount = isset( $detail->amount ) ? $detail->amount : 0;
			if ( $amount ) {
				wptravel_update_payment_status( $booking_id, $amount, 'paid', $detail, sprintf( '_%s_args', $payment_gateway ), $new_payment_id );

				$_SESSION['used-razorpay'] = true;
			}
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}
	}
endif;
