<?php
/**
 * Stripe Functions.
 *
 * @package wp-travel-stripe/inc/
 */

/**
 * Generate Payment Response
 *
 * @param object $intent
 * @return void
 */
function wp_travel_generate_payment_response( $intent ) {
	// Note: Stripe sdk Update/migrate status from requires_source_action to requires_action.
	if ( ( 'requires_source_action' === $intent->status || 'requires_action' === $intent->status ) &&
	'use_stripe_sdk' === $intent->next_action->type ) {
		// Tell the client to handle the action.
		echo wp_json_encode(
			array(
				'id'                           => $intent->id,
				'requires_action'              => true,
				'payment_intent_client_secret' => $intent->client_secret,
			)
		);
	} elseif ( 'succeeded' === $intent->status ) {
		// The payment didn’t need any additional actions and completed!
		// Handle post-payment fulfillment.
		echo wp_json_encode(
			array(
				'id'                           => $intent->id,
				'success'                      => true,
				'payment_intent_client_secret' => $intent->client_secret,
			)
		);
	} else {
		// Invalid status.
		http_response_code( 500 );
		echo wp_json_encode( array( 'error' => 'Invalid PaymentIntent status' ) );
	}
	die;
}

/**
 * Ajax Endpoint for action: payment_intents.
 *
 * @return void
 */
function wp_travel_stripe_payment_intent() {
	if ( function_exists( 'wptravel_get_settings' ) ) {
		$settings = wptravel_get_settings();
	} else {
		$settings = wp_travel_get_settings();
	}
	$secret_key = $settings['stripe_secret_key'];

	if ( function_exists( 'wptravel_test_mode' ) ) {
		$test_mode = wptravel_test_mode();
	} else {
		$test_mode = wp_travel_test_mode();
	}
	if ( $test_mode ) {
		$secret_key = $settings['stripe_test_secret_key'];
	}
	$currency_code           = ( isset( $settings['currency'] ) ) ? $settings['currency'] : 'USD';
	// $stripe_enable_3d_secure = ( isset( $settings['stripe_enable_3d_secure'] ) ) ? $settings['stripe_enable_3d_secure'] : 'no';
	

	\Stripe\Stripe::setApiKey( $secret_key );
	header( 'Content-Type: application/json' );

	// Retrieve json from POST body.
	$json_str = file_get_contents( 'php://input' );
	$json_obj = json_decode( $json_str );
	$intent   = null;
	try {
		if ( isset( $json_obj->payment_method_id ) ) {
			// Create the PaymentIntent.

			// since v6.4.0 - https://stripe.com/docs/upgrades#2023-08-16
			$stripe_args = array(
				'payment_method'      => $json_obj->payment_method_id,
				'amount'              => wp_travel_stripe_convert_amount( $json_obj->data->amount, $json_obj->data->currency ),
				'currency'            => strtolower( $json_obj->data->currency ),
				// 'confirmation_method' => 'manual',
				// since v6.4.0
				'automatic_payment_methods' => array(
					'enabled' => true,
					'allow_redirects' => 'never'
				),
				'confirm'             => true,
				'return_url' => get_site_url(),
			);

			// if ( 'yes' === $stripe_enable_3d_secure ) {
			// 	// $stripe_args['payment_method_options']['card']['request_three_d_secure'] = 'any';
			// }
			$intent = \Stripe\PaymentIntent::create( $stripe_args );
		}
		if ( isset( $json_obj->payment_intent_id ) ) {
			$intent = \Stripe\PaymentIntent::retrieve(
				$json_obj->payment_intent_id
			);
			$intent->confirm();
		}
		wp_travel_generate_payment_response( $intent );
	} catch ( \Stripe\Error\Base $e ) {
		// Display error on client.
		echo wp_json_encode(
			array(
				'error' => $e->getMessage(),
			)
		);
		die;
	}
}

/**
 * Charge the order, publish the payment, create the user, email the user
 */
function wp_travel_stripe_charge() {

	if ( ! empty( $_POST['gateway'] ) && 'stripe' == $_POST['gateway'] ) {

		ini_set( 'display_errors', 0 );
		// error_log( serialize( $_POST ) );
		if ( function_exists( 'wptravel_get_settings' ) ) {
			$settings = wptravel_get_settings();
		} else {
			$settings = wp_travel_get_settings();
		}
		// Settings api key.
		$publishable_key = $settings['stripe_publishable_key'];
		$secret_key      = $settings['stripe_secret_key'];

		if ( function_exists( 'wptravel_test_mode' ) ) {
			$test_mode = wptravel_test_mode();
		} else {
			$test_mode = wp_travel_test_mode();
		}
		if ( $test_mode ) {
			$publishable_key = $settings['stripe_test_publishable_key'];
			$secret_key      = $settings['stripe_test_secret_key'];
		}
		$stripe = array(
			'publishable_key' => $publishable_key,
			'secret_key'      => $secret_key,
		);

		\Stripe\Stripe::setApiKey( $stripe['secret_key'] );
		// Settings api key Ends.

		$token = $_POST['token']['id'];
		$email = $_POST['token']['email'];

		$customer = \Stripe\Customer::create(
			array(
				'email'  => $email,
				'source' => $token,
			)
		);

		$currency_code = ( isset( $settings['currency'] ) ) ? $settings['currency'] : 'USD';
		$total         = $_POST['total'];

		try {
			$session = \Stripe\Checkout\Session::create(
				[
					'payment_method_types' => [ 'card' ],
					'line_items'           => [
						[
							'name'        => 'T-shirt',
							'description' => 'Comfortable cotton t-shirt',
							'images'      => [ 'https://example.com/t-shirt.png' ],
							'amount'      => 500,
							'currency'    => 'usd',
							'quantity'    => 1,
						],
					],
					'success_url'          => 'https://example.com/success',
					'cancel_url'           => 'https://example.com/cancel',
				]
			);
			// Charge the customer
			// $charge = \Stripe\Charge::create(array(
			// 'customer' => $customer->id,
			// 'amount'   => wp_travel_stripe_convert_amount( $total, $currency_code ),
			// 'currency' => strtolower( $currency_code ),
			// 'source'   => $customer->default_source,
			// ));
			$status  = true;
			$message = $session;

		} catch ( Exception $e ) {
			$message = $e->getMessage();
			$status  = false;
			$charge  = '';
		}

		wp_send_json(
			array(
				'status'  => $status,
				// 'charge'  => json_encode( $charge ),
				'message' => $message,
			)
		);
		die;

	}
}



/**
 * With decimal currencies like USD
 * $5 must be multiplied by 100
 * to give a cent amount of "500".
 * But for zero-decimal currencies (like JPY),
 * ¥500 is just ¥500.
 *
 * The following currencies should be excluded from multiplication:
 * BIF, CLP, DJF, GNF, JPY, KMF, KRW, MGA, PYG, RWF, VND, VUV, XAF, XOF, XPF
 *
 * https://support.stripe.com/questions/which-zero-decimal-currencies-does-stripe-support
 *
 * @var int (total)
 * @return int (converted total)
 */
function wp_travel_stripe_convert_amount( $total = null, $currency = null ) {

	if ( ! $currency ) {
		$currency = 'USD';
	}
	$zero_decimal_currencies = array( 'BIF', 'CLP', 'DJF', 'GNF', 'JPY', 'KMF', 'KRW', 'MGA', 'PYG', 'RWF', 'VND', 'VUV', 'XAF', 'XOF', 'XPF' );

	if ( in_array( $currency, $zero_decimal_currencies ) ) {
		$new_total = intval( $total );
	} else {
		$new_total = round( (float) $total, 2 ) * 100;
	}
	return $new_total;
}

/**
 * Response amount from stripe payment intent.
 * With decimal currencies like USD
 * $5 (500) must be divided by 100
 * to get actual $ value as $5.
 * But for zero-decimal currencies (like JPY),
 * ¥500 is just ¥500.
 *
 * The following currencies should be excluded from multiplication:
 * BIF, CLP, DJF, GNF, JPY, KMF, KRW, MGA, PYG, RWF, VND, VUV, XAF, XOF, XPF
 *
 * https://support.stripe.com/questions/which-zero-decimal-currencies-does-stripe-support
 *
 * @var int (total)
 * @return int (converted total)
 */
function wp_travel_stripe_revert_amount( $total = null, $currency = null ) {

	if ( ! $currency ) {
		$currency = 'USD';
	}
	$zero_decimal_currencies = array( 'BIF', 'CLP', 'DJF', 'GNF', 'JPY', 'KMF', 'KRW', 'MGA', 'PYG', 'RWF', 'VND', 'VUV', 'XAF', 'XOF', 'XPF' );

	if ( in_array( $currency, $zero_decimal_currencies ) ) {
		$new_total = intval( $total );
	} else {
		$new_total = round( (float) $total, 2 ) / 100;
	}
	return $new_total;
}

function wp_travel_stripe_booking_message( $message ) {
	unset( $_SESSION['used-stripe'] );
	$message = esc_html__( "We've received your booking and payment details. We'll contact you soon.", 'wp-travel-pro' );
	return $message;
}