<?php
/**
 * Stripe Functions.
 *
 * @package wp-travel-stripe/inc/
 */

/**
 * Generate Payment Response
 *
 * @param object $intent
 * @return void
 */

function wp_travel_generate_payment_response( $intent ) {
	// Note: Stripe sdk Update/migrate status from requires_source_action to requires_action.

	if ( $intent['status'] == 'requires_payment_method'  ) {
		// Tell the client to handle the action.
		echo wp_json_encode(
			array(
				'id'                           => $intent->id,
				'requires_action'              => true,
				'payment_intent_client_secret' => $intent->client_secret,
			)
		);
	} else {
		// Invalid status.
		http_response_code( 500 );
		echo wp_json_encode( array( 'error' => 'Invalid PaymentIntent status' ) );
	}
	die;
}

/**
 * Ajax Endpoint for action: payment_intents.
 *
 * @return void
 */
function wp_travel_stripe_payment_intent() {
	if ( function_exists( 'wptravel_get_settings' ) ) {
		$settings = wptravel_get_settings();
	} else {
		$settings = wp_travel_get_settings();
	}
	$secret_key = $settings['stripe_secret_key'];

	if ( function_exists( 'wptravel_test_mode' ) ) {
		$test_mode = wptravel_test_mode();
	} else {
		$test_mode = wp_travel_test_mode();
	}
	if ( $test_mode ) {
		$secret_key = $settings['stripe_test_secret_key'];
	}
	$currency_code           = ( isset( $settings['currency'] ) ) ? $settings['currency'] : 'USD';
	// $stripe_enable_3d_secure = ( isset( $settings['stripe_enable_3d_secure'] ) ) ? $settings['stripe_enable_3d_secure'] : 'no';
	

	\Stripe\Stripe::setApiKey( $secret_key );
	header( 'Content-Type: application/json' );

	// Retrieve json from POST body.
	$json_str = file_get_contents( 'php://input' );
	$json_obj = json_decode( $json_str );
	$intent   = null;

	global $wpdb;
	$latest_post_id = $wpdb->get_var("SELECT ID FROM $wpdb->posts ORDER BY ID DESC LIMIT 1");

	try {
		// Create the PaymentIntent.

		$customer = \Stripe\Customer::create([
			'name' => $json_obj->data->customer_name,
			'description' => __( 'WP Travel Customer', 'wp-travel-pro' ),
			'email' => $json_obj->data->customer_email,
			'phone' => $json_obj->data->customer_contact_number,
			'address'     => [
				'country' => $json_obj->data->customer_country,
			],
		]);

		// since v6.4.0 - https://stripe.com/docs/upgrades#2023-08-16
		$stripe_args = array(
			// 'payment_method'      => $json_obj->payment_method_id,
			'amount'              => wp_travel_stripe_convert_amount( $json_obj->data->amount, $json_obj->data->currency ),
			'currency'            => strtolower( $json_obj->data->currency ),
			'customer' => $customer->id,
			// 'confirmation_method' => 'manual',
			// since v6.4.0
			'automatic_payment_methods' => array(
				'enabled' => true,
				// 'allow_redirects' => 'never'
			),
			'metadata'    => [
				'Booking Id' => (int)$latest_post_id + 1,
				'Booked Trip'     => $json_obj->data->trip_title,
			],

		);

		// if ( 'yes' === $stripe_enable_3d_secure ) {
		// 	// $stripe_args['payment_method_options']['card']['request_three_d_secure'] = 'any';
		// }
		$intent = \Stripe\PaymentIntent::create( $stripe_args );
	

		wp_travel_generate_payment_response( $intent );
	} catch ( \Stripe\Error\Base $e ) {
		// Display error on client.
		echo wp_json_encode(
			array(
				'error' => $e->getMessage(),
			)
		);
		die;
	}
}


function wp_travel_stripe_convert_amount( $total = null, $currency = null ) {

	if ( ! $currency ) {
		$currency = 'USD';
	}
	$zero_decimal_currencies = array( 'BIF', 'CLP', 'DJF', 'GNF', 'JPY', 'KMF', 'KRW', 'MGA', 'PYG', 'RWF', 'VND', 'VUV', 'XAF', 'XOF', 'XPF' );

	if ( in_array( $currency, $zero_decimal_currencies ) ) {
		$new_total = intval( $total );
	} else {
		$new_total = round( (float) $total, 2 ) * 100;
	}
	return $new_total;
}

function wp_travel_stripe_revert_amount( $total = null, $currency = null ) {

	if ( ! $currency ) {
		$currency = 'USD';
	}
	$zero_decimal_currencies = array( 'BIF', 'CLP', 'DJF', 'GNF', 'JPY', 'KMF', 'KRW', 'MGA', 'PYG', 'RWF', 'VND', 'VUV', 'XAF', 'XOF', 'XPF' );

	if ( in_array( $currency, $zero_decimal_currencies ) ) {
		$new_total = intval( $total );
	} else {
		$new_total = round( (float) $total, 2 ) / 100;
	}
	return $new_total;
}