<?php
/**
 * WP Travel Stripe iDEAL Checkout Core Class.
 *
 * @package wp-travel-stripe-ideal-checkout-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

// WP Travel Stripe iDEAL Checkout core.
if ( ! class_exists( 'WP_Travel_Stripe_Ideal_Checkout_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Stripe_Ideal_Checkout_Core {

		const WP_TRAVEL_STRIPE_IDEAL_CHECKOUT_HANDLE = 'wp_travel_stripe_ideal_checkout_';

		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var object WP Travel Stripe iDEAL Checkout Core
		 * @since 1.0.0
		 */
		protected static $instance = null;

		/**
		 * Main WP_Travel_Stripe_Ideal_Checkout_Core Instance.
		 * Ensures only one instance of WP_Travel_Stripe_Ideal_Checkout_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Stripe_Ideal_Checkout_Core()
		 * @return WP_Travel_Stripe_Ideal_Checkout_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			self::includes();
			add_action( 'wp_enqueue_scripts', array( __CLASS__, 'frontend_assets' ), 20 );
			add_action( 'admin_enqueue_scripts', array( __CLASS__, 'admin_assets' ), 20 );

			// Payment Gateway list.
			add_filter( 'wp_travel_payment_gateway_lists', 'wp_travel_stripe_ideal_gateway' );
			add_filter( 'wp_travel_premium_addons_list', 'wp_travel_stripe_ideal_addons' );

			if ( self::is_enabled() ) {
				add_filter( 'wptravel_frontend_data', 'wp_travel_stripe_ideal_add_vars', 10, 2 );
			}

			add_action( 'wp_travel_action_after_payment_info_field', array( __CLASS__, 'add_extra_fields' ) );
			add_action( 'wp_travel_dashboard_booking_after_detail', array( __CLASS__, 'add_extra_fields' ), 15, 2 );

			if ( self::is_the_gateway() ) {
				// This will call after payement completed.
				add_action( 'wp_travel_after_frontend_booking_save', array( __CLASS__, 'set_up_payment_options' ), 20 );
				add_action( 'wp_travel_after_partial_payment_complete', array( __CLASS__, 'set_up_partial_payment' ), 20 );
			}
			isset( $_SESSION['used-stripe-ideal'] ) && $_SESSION['used-stripe-ideal'] && add_filter( 'wp_travel_booked_message', 'wp_travel_stripe_ideal_booking_message', 20 );

		}

		/**
		 * Add Error div.
		 *
		 * @param int   $booking_id
		 * @param mixed $details
		 * @return void
		 */
		public static function add_extra_fields( $booking_id = null, $details = null ) {
			?>
			<div id="stripe-ideal-errors"></div>
			<?php
		}

		/**
		 * Determine if booking used express checkout.
		 */
		private static function is_the_gateway() {
			$settings = wptravel_get_settings();
			if( isset( $settings['selected_booking_option'] ) && count( $settings['selected_booking_option'] ) == 1  && $settings['selected_booking_option'][0] == 'booking-with-payment' ){
				return 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'stripe_ideal' === $_REQUEST['wp_travel_payment_gateway'];
			}else{
				return isset( $_POST['wp_travel_booking_option'] ) && 'booking_with_payment' === $_POST['wp_travel_booking_option'] && 'POST' === $_SERVER['REQUEST_METHOD'] && array_key_exists( 'wp_travel_payment_gateway', $_REQUEST ) && 'stripe_ideal' === $_REQUEST['wp_travel_payment_gateway'];
			}
			
		}

		/**
		 * Determine if Stripe checkout is enabled.
		 */
		private static function is_enabled() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}
			return array_key_exists( 'payment_option_stripe_ideal', $settings ) && 'yes' === $settings['payment_option_stripe_ideal'];
		}

		/**
		 * Determing is Stripe checkout is disabled.
		 */
		private static function is_disabled() {
			return ! self::is_enabled();
		}

		/**
		 * Stripe Frontend assets.
		 */
		public static function frontend_assets() {

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			if ( function_exists( 'wptravel_can_load_payment_scripts' ) ) {
				$load_payment_script = wptravel_can_load_payment_scripts();
			} else {
				$load_payment_script = wp_travel_can_load_payment_scripts();
			}

			if ( $load_payment_script && self::is_enabled() && isset( $settings['enable_one_page_booking'] ) && $settings['enable_one_page_booking'] == "" ) {

				$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
				// Styles.
				$dependencies   = array( 'jquery', 'wp-travel-payment-frontend-script' );
				$dependencies[] = 'wp-travel-payment-frontend-script';
				wp_enqueue_style( 'wp-travel-stripe-ideal-custom', self::$plugin_url . 'assets/css/wp-travel-stripe-ideal-view' . $suffix . '.css', '', self::$version );
				wp_enqueue_script( 'wp_travel_stripe_checkout_js', 'https://js.stripe.com/v3/', array( 'jquery' ), self::$version, true );

				wp_register_script( 'wp-travel-stripe-ideal-script', self::$plugin_url . 'assets/js/wp-travel-stripe-ideal-view' . $suffix . '.js', $dependencies, self::$version, true );

				wp_enqueue_script( 'wp-travel-stripe-ideal-script' );
			}
			global $post;
			// wp-travel-authorize-net-checkout-frontend
			if ( get_post_type() == 'itineraries' && isset( $settings['enable_one_page_booking'] ) && $settings['enable_one_page_booking'] == 1 ) {
				wp_register_script( 'wp_travel_stripe_ideal_booking_js', 'https://js.stripe.com/v3/', array( 'jquery' ), self::$version, true );
				$depend = array( 'wp-travel-frontend-booking-widget', 'wp-travel-payment-frontend-script' );
				wp_register_script( 'wp-travel-stripe-ideal-frontend-script', self::$plugin_url . 'app/build/ideal-checkout-on-page-nooking.js', $depend, self::$version, true );
				wp_enqueue_script( 'wp-travel-stripe-ideal-frontend-script' );
			}
		}

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {
			$suffix         = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$screen         = get_current_screen();
			$allowed_screen = array( 'edit-itineraries', WP_TRAVEL_POST_TYPE, 'itineraries_page_settings', 'itinerary-booking', 'itinerary-booking_page_settings' );

			$switch_to_react = true; // force enable.
			if ( function_exists( 'wptravel_is_react_version_enabled' ) ) {
				$switch_to_react = wptravel_is_react_version_enabled();  // force enable but has filter to include backward compatibility for now.
			}
			// settings_screen.
			if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_STRIPE_IDEAL_CHECKOUT_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
			if ( ! $switch_to_react ) {
				if ( in_array( $screen->id, $allowed_screen ) ) {
					wp_enqueue_style( 'wp-travel-stripe-ideal', self::$plugin_url . 'assets/css/wp-travel-stripe-ideal' . $suffix . '.css', array(), self::$version );
				}
			}
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			if ( ! class_exists( 'Stripe\Stripe' ) ) {
				require_once sprintf( '%s/vendor/stripe/init.php', self::$abspath );
			}

			// if (!class_exists(\Stripe\Stripe::class)) {
			// 	require_once __DIR__ . '/new-vendor/autoload.php'; // Adjust path as necessary
			// }

			include sprintf( '%s/inc/functions.php', self::$abspath );
			include sprintf( '%s/inc/wp-travel-stripe-ideal-function.php', self::$abspath );
			if ( self::is_request( 'admin' ) ) {
				include sprintf( '%s/inc/admin/settings.php', self::$abspath );
			}
		}

		/**
		 * Sets up payment options
		 *
		 * @param string $booking_id ID of booking.
		 * @return void
		 */
		public static function set_up_payment_options( $booking_id ) {
			
			if ( self::is_disabled() || ! self::is_the_gateway() || ! $booking_id || ! isset( $_POST['wp_travel_book_now'] ) ) {
				return;
			}

			do_action( 'wt_before_payment_process', $booking_id );

			$json = sanitize_text_field( wp_unslash( $_REQUEST['payment_details'] ) );

			$detail = json_decode( $json );
			$amount = isset( $detail->amount ) ? $detail->amount : 0;

			if ( $amount ) {
				$amount = number_format( $amount / 100, 2, '.', '' );
			}
			$payment_id     = get_post_meta( $booking_id, 'wp_travel_payment_id', true );
			$payment_method = 'stripe_ideal';
			update_post_meta( $payment_id, 'wp_travel_payment_gateway', $payment_method );

			wptravel_update_payment_status(
				$booking_id,
				$amount,
				$detail->status,
				array(
					'source_id' => $detail->id,
					'status'    => 'paid',
				),
				sprintf( '_%s_args', $payment_method ),
				$payment_id
			);
			set_transient( 'bid_' . $detail->id, $booking_id, 12 * HOUR_IN_SECONDS );
			set_transient( 'pid_' . $detail->id, $payment_id, 12 * HOUR_IN_SECONDS );
			wp_redirect( $detail->redirect->url );
			exit;
		}

		/**
		 * Sets up payment options
		 *
		 * @return void
		 */
		public static function set_up_partial_payment() {

			if ( self::is_disabled() || ! self::is_the_gateway() || ! isset( $_POST['complete_partial_payment'] ) || ! isset( $_POST['wp_travel_booking_id'] ) ) {
				return;
			}

			$payment_gateway = sanitize_text_field( wp_unslash( $_POST['wp_travel_payment_gateway'] ) );
			$booking_id      = sanitize_text_field( wp_unslash( $_POST['wp_travel_booking_id'] ) );
			// get previous payment ids.
			$payment_id = get_post_meta( $booking_id, 'wp_travel_payment_id', true );

			/**
			 * This will return the current payment id to insert payment meta for the current payment.
			 *
			 * @hooked wptravel_update_payment_post from partial payment which will also update booking option [booking_only | booking_with_payment ]
			 *
			 * @since 4.0.2
			 */
			$new_payment_id = apply_filters( 'wptravel_before_insert_partial_payment', $payment_id, $booking_id, $payment_gateway );

			$meta = get_post_meta( $booking_id, 'wp_travel_payment_id', true );

			$json = sanitize_text_field( wp_unslash( $_POST['payment_details'] ) );

			$detail = json_decode( $json );
			$amount = isset( $detail->amount ) ? $detail->amount : 0;

			if ( $amount ) {
				$amount = number_format( $amount / 100, 2, '.', '' );
				wptravel_update_payment_status(
					$booking_id,
					$amount,
					$detail->status,
					array(
						'source_id' => $detail->id,
						'status'    => 'paid',
					),
					sprintf( '_%s_args', $payment_gateway ),
					$new_payment_id
				);
				set_transient( 'bid_' . $detail->id, $booking_id, 12 * HOUR_IN_SECONDS );
				set_transient( 'pid_' . $detail->id, $new_payment_id, 12 * HOUR_IN_SECONDS );
				wp_redirect( $detail->redirect->url );
				exit;
			}
		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}
	}
endif;