<?php 
namespace WP_Travel_Travel_Guide_Core;
use WP_REST_Server;
use WP_REST_Response;
use WP_REST_Request;

/**
 * Base class for registering api routes
 * 
 * @see https://developer.wordpress.org/reference/functions/register_rest_route/
 * @since WP Travel Travel Guide 1.0
 */

if( !class_exists( 'Base_Api' ) ){
	class Base_Api{


		/**
		 * defining API version
		 *
		 */
		protected $version = 'v1';

		/**
		 * defining API End Ponit name
		 *
		 */
		protected $name = "travel-guide";

		protected static $instance = null;


		/**
		 * defining response status
		 *
		 */
		protected $status = 200;

		protected $data = array(
			'data'    => array(),
			'message' => '',
			'validation_errors' => false
		);


		/**
		 * defining response methods
		 *
		 */
		protected $methods = array(
			'get'    => WP_REST_Server::READABLE,
			'post'   => WP_REST_Server::CREATABLE,
			'put'    => WP_REST_Server::EDITABLE,
			'delete' => WP_REST_Server::DELETABLE
		);

		public function __construct(){

			/**
			 * defining API End Ponit name
			 *
			 */
			add_action( 'rest_api_init', array( $this, 'register' ) );
		}


		/**
		 * Register API Route
		 *
		 */
		public function register(){

			$routes = apply_filters( 'wptravel_travel_guide_rest_routes', $this->get_routes(), $this );

			foreach( $routes as $args ){
				$route = $this->get_route( $args[ 'route' ] );
				$args[ 'methods' ] = $this->get_method( $args[ 'method' ] );
				$args[ 'permission_callback' ] = array( $this, 'check_permission' );

				unset( $args[ 'route' ] );
				register_rest_route( 
					$this->get_namespace(), 
					$route, 
					$args 
				);
			}

			register_rest_route( 'wp-travel/v1', '/trip/(?P<id>\d+)', array(
		        'methods' => 'GET',
		        'permission_callback' => '__return_true',
		        'callback' => array( $this, 'get_trip_guide' ),
		    ));
		}


		/**
		 * Get all trip guide - callback funtion
		 *
		 */
		public function get_trip_guide( WP_REST_Request $request ){

			$selected_trip_guide_ids = get_post_meta( $request->get_param( 'id' ), 'selected_guides', true );
			$guide_data = array();
			
			if ( $selected_trip_guide_ids ) {
				foreach ( $selected_trip_guide_ids as $guide_id) {
					$get_guide_data = get_userdata( ( int ) $guide_id )->data;
					if( $get_guide_data->ID !== null ){
						$data['guide_id'] = $get_guide_data->ID;
						$data['fullname'] = $get_guide_data->display_name;
						$data['email'] = $get_guide_data->user_email;
						$data['user_phone'] = get_user_meta( $data['guide_id'], 'user_phone', true );
						$data['city'] = get_user_meta( $data['guide_id'], 'city', true );
						$data['country'] = get_user_meta( $data['guide_id'], 'country', true );
						$data[ 'guide_url' ]  = get_post( get_posts( array( 'name' => $get_guide_data->user_login, 'post_type' => 'travel-guide', ) )[0]->ID )->guid ;
						$data['guide_image'] = !empty( get_user_meta( $data['guide_id'], 'profile_picture', true ) ) ? get_user_meta( $data['guide_id'], 'profile_picture', true )['url'] : WP_Travel_Travel_Guide_Core_URL . 'assets/wp-travel-placeholder.png';
						$data['guide_experience'] = !empty( get_user_meta( $data['guide_id'], 'join_year', true ) ) ? date( 'Y' ) - get_user_meta( $data['guide_id'], 'join_year', true ): 0;
						
						if ( $data['guide_experience'] > 0 ) {
							$data['guide_experience'] = $data['guide_experience'].__( 'years', 'wp-travel-pro' );
						}else{
							$data['guide_experience'] = __( 'Less than a year', 'wp-travel-pro' );
						}

						$data['guide_location'] = !empty( get_user_meta( $data['guide_id'], 'city', true ) ) ? get_user_meta( $data['guide_id'], 'city', true ): '';

						$data['guide_language'] = !empty( get_user_meta( $data['guide_id'], 'language', true ) ) ? get_user_meta( $data['guide_id'], 'language', true ): '';

						$data['guide_description'] = !empty( get_user_meta( $data['guide_id'], 'description', true ) ) ? get_user_meta( $data['guide_id'], 'description', true ): '';

						$data['facebook_link'] = !empty( get_user_meta( $data['guide_id'], 'facebook_link', true ) ) ? get_user_meta( $data['guide_id'], 'facebook_link', true ): '';

						$data['instagram_link'] = !empty( get_user_meta( $data['guide_id'], 'instagram_link', true ) ) ? get_user_meta( $data['guide_id'], 'instagram_link', true ): '';

						$data['twitter_link'] = !empty( get_user_meta( $data['guide_id'], 'twitter_link', true ) ) ? get_user_meta( $data['guide_id'], 'twitter_link', true ): '';

						$data['linkedin_link'] = !empty( get_user_meta( $data['guide_id'], 'linkedin_link', true ) ) ? get_user_meta( $data['guide_id'], 'linkedin_link', true ): '';

						$data['youtube_link'] = !empty( get_user_meta( $data['guide_id'], 'youtube_link', true ) ) ? get_user_meta( $data['guide_id'], 'youtube_link', true ): '';

						$data['tiktok_link'] = !empty( get_user_meta( $data['guide_id'], 'tiktok_link', true ) ) ? get_user_meta( $data['guide_id'], 'tiktok_link', true ): '';

						$get_reviews = get_comments( array( 'post_id' => get_posts( array( 'name' => $get_guide_data->user_login, 'post_type' => 'travel-guide', ) )[0]->ID ) ); 
						$review_count = count( get_comments( array( 'post_id' => get_posts( array( 'name' => $get_guide_data->user_login, 'post_type' => 'travel-guide', ) )[0]->ID ) ) ); 
						$get_rating = 0;

						foreach ( $get_reviews as $review ) {
						$get_rating = $get_rating + ( int )get_comment_meta( ( int )$review->comment_ID, '_wp_travel_rating' )[0];
						}

						$data[ 'avg_rating' ] = ( $review_count < 1 ) ? 0 : intval( $get_rating/$review_count );
						
						array_push( $guide_data, $data );
					}
					
				}
				return $guide_data;
			}else{
				return $guide_data;
			}			
			
		}

		/**
		 * namespace
		 *
		 * @return string
		 */
		public function get_namespace(){
			return 'wp-travel/' . $this->version;
		}


		/**
		 * request method
		 *
		 * @return string
		 */
		public function get_method( $m ){
			return $this->methods[ $m ];
		}


		/**
		 * route
		 *
		 * @return string
		 */
		public function get_route( $route ){
			return '/' . $this->name . '/' . $route;
		}


		/**
		 * check if numeric
		 *
		 * @return boolean
		 */
		public function is_numeric( $num ){
			return is_numeric( $num );
		}


		/**
		 * check permission
		 *
		 * @return boolean
		 */
		public function check_permission(){ return true;
			return current_user_can( 'edit_others_posts' );
		}


		/**
		 * check validation
		 *
		 * @return boolean
		 */
		public function set_validation_errors( $errors ){
			$this->data[ 'validation_errors' ] = $errors;
		}


		/**
		 * set message
		 *
		 * @return string
		 */
		public function set_message( $message ){
			$this->data[ 'message' ] = $message;
		}


		/**
		 * set status
		 *
		 * @return boolean
		 */
		public function set_status( $status ){
			$this->data[ 'status' ] = $status;
		}


		/**
		 * get response 
		 *
		 * @return boolean
		 */
		public function get_response( $headers = array() ){
			return new WP_REST_Response( $this->data, $this->status, $headers );
		}


		/**
		 * get data 
		 *
		 * @return array
		 */
		public function set_data( $data ){
			$this->data[ 'data' ] = $data;
		}
	}
}