<?php
/**
 * WP Travel Utilities Core Class.
 *
 * @package wp-travel-utilities-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Utilities core.
if ( ! class_exists( 'WP_Travel_Utilities_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Utilities_Core {

		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Utilities Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Utilities_Core Instance.
		 * Ensures only one instance of WP_Travel_Utilities_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Utilities_Core()
		 * @return WP_Travel_Utilities_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_utilities = isset( $settings['show_wp_travel_utilities'] ) ? $settings['show_wp_travel_utilities'] : 'yes';

			// add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Utilities_Core', 'settings_fields' ) ); // moved field in helpers/settings.php
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Utilities_Core', 'plugin_action' ) );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Utilities_Core', 'admin_assets' ) );

			// Settings Helpers
			include sprintf( '%s/core/helpers/settings.php', self::$abspath );
			// if ( 'yes' !== $enable_utilities ) {
			// add_action( 'wptravel_trip_cart_checkout_tab_content', array( 'WP_Travel_Utilities_Core', 'plugin_disabled_notice' ) );
			// add_action( 'wp_travel_settings_tab_cart_checkout_fields', array( 'WP_Travel_Utilities_Core', 'plugin_disabled_notice' ) );
			// add_action( 'wp_travel_custom_global_tabs', array( 'WP_Travel_Utilities_Core', 'plugin_disabled_notice' ) );
			// add_action( 'wp_travel_settings_tab_faq_fields', array( 'WP_Travel_Utilities_Core', 'plugin_disabled_notice' ) );

			// return;
			// }

			// Global tab support custom tab.
			add_filter( 'wp_travel_is_custom_tabs_support_enabled', '__return_true' );

			// itinerary tabs support custom tabs.
			add_filter( 'wp_travel_custom_itinerary_tabs', '__return_true' );

			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Utilities_Core', 'frontend_assets' ) );

			add_action( 'init', array( 'WP_Travel_Utilities_Core', 'includes' ) );

			add_filter( 'wp_travel_enable_custom_tab', array( 'WP_Travel_Utilities_Core', 'enable_custom_tab' ) );

			// Add custom pricing option in dropdown list.
			add_filter( 'wp_travel_pricing_option_list', array( 'WP_Travel_Utilities_Core', 'custom_pricing_option' ) );
			// Markup.
			// add_action( 'wp_travel_itinerary_price_tab_table_last_row', array( 'WP_Travel_Utilities_Core', 'custom_pricing_option_content' ) );
			// Save meta.
			add_filter( 'wp_travel_save_trip_metas', array( 'WP_Travel_Utilities_Core', 'custom_pricing_option_content_save' ), 10, 2 );
			// Display custom Markup in frontend. [wp_travel_{$pricing_option_type}_options_list].
			// add_action( 'wp_travel_custom-booking_options_list', array( 'WP_Travel_Utilities_Core', 'content_pricing_option' ) );

			/**
			 * Utilities Data Filter for WP Travel v4.0.0
			 *
			 * @since 1.0.3
			 */
			add_filter( 'wp_travel_trip_data', array( __CLASS__, 'add_rest_data' ), 11, 2 );
			add_action( 'wp_travel_update_trip_data', array( __CLASS__, 'save_rest_data' ), 11, 2 );

			// Inventory Helpers
			include sprintf( '%s/core/helpers/utilities.php', self::$abspath );

			/**
			 * Filter
			 *
			 * @since wptravel 4.0.6
			 */
			add_filter(
				'wp-travel-script-handles',
				function( $handles ) {
					$handles[] = 'wp-travel-utilities-front-end-js';
					return $handles;
				}
			);
			/**
			 * To enable add to cart while activate multiple checkout
			 * @since wptravel 7.1
			 */
			add_action( 'wp_travel_add_to_cart_system', function ( $args )
			{ 
				$settings       = wptravel_get_settings();
				$multiple_checkout_enable  = isset( $settings['enable_multiple_checkout'] ) ? $settings['enable_multiple_checkout'] : 'no';
				return $multiple_checkout_enable == 'yes' ? true : false; 
			} );
		}

		/**
		 * Meta fields added by the addon.
		 *
		 * @since react // TODO: Update version on release.
		 */
		private static function get_trip_meta_fields( $trip_id ) {
			if ( function_exists( 'wptravel_get_trip_code' ) ) {
				$trip_code = wptravel_get_trip_code( $trip_id );
			} else {
				$trip_code = wp_travel_get_trip_code( $trip_id );
			}
			return array(
				/**
				 * Dates and Prices Tab.
				 */
				// 'wp_travel_custom_booking_type'      => array(
				// 'default' => 'custom-link',
				// 'type'    => 'string',
				// 'options' => array( 'custom-link', 'custom-form' ),
				// ),
				// 'wp_travel_custom_booking_link' => array(
				// 'default' => '',
				// 'type'    => 'string',
				// ),
				// 'wp_travel_custom_booking_link_text' => array(
				// 'default' => __( 'Book Now' ),
				// 'type'    => 'string',
				// ),
				// 'wp_travel_custom_booking_link_open_in_new_tab' => array(
				// 'default' => '0',
				// 'type'    => 'string',
				// 'options' => array( '1', '0' ),
				// ),
				// 'wp_travel_custom_booking_form'      => array(
				// 'default' => '',
				// 'type'    => 'string',
				// ),
				/**
				 * Cart and Checkout Tab.
				 */
				'wp_travel_trip_checkout_page_info'       => array(
					'default' => '',
					'type'    => 'string',
				),
				'wp_travel_trip_checkout_page_info_label' => array(
					'default' => '',
					'type'    => 'string',
				),
				/**
				 * Faqs Tab.
				 */
				'wp_travel_utils_use_global_faq_for_trip' => array(
					'default' => 'no',
					'type'    => 'string',
					'options' => array( 'yes', 'no' ),
				),
				'wp_travel_utils_use_trip_faq_for_trip'   => array(
					'default' => 'no',
					'type'    => 'string',
					'options' => array( 'yes', 'no' ),
				),
				/**
				 * Tabs tab.
				 */
				'wp_travel_itinerary_custom_tab_cnt_'     => array(
					'default' => array(),
					'type'    => 'array',
				),
				/**
				 * Itinerary Tab.
				 */
				'wp_travel_trip_code'                     => array(
					'default' => $trip_code,
					'type'    => 'string',
				),
			);
		}

		/**
		 * Adds Utilities data new rest api.
		 *
		 * @since react
		 */
		public static function add_rest_data( $trip_data, $trip_id = 0 ) {
			$data       = array();
			$field_name = 'utilities';
			foreach ( (array) self::get_trip_meta_fields( $trip_id ) as $meta_key => $field_schema ) {
				$meta_value = get_post_meta( $trip_id, $meta_key, true );
				if ( isset( $field_schema['options'] ) && ! in_array( $meta_value, $field_schema['options'] ) ) {
					$meta_value = $field_schema['default'];
				}
				$data[ $meta_key ] = ! empty( $meta_value ) && $field_schema['type'] === gettype( $meta_value ) ? $meta_value : $field_schema['default'];
				// Map Tab Data.
				if ( 'wp_travel_itinerary_custom_tab_cnt_' === $meta_key && ! empty( $data[ $meta_key ] ) ) {
					// error_log( print_r( $data[ $meta_key ], true ) );
					$tabs = array();
					foreach ( $data[ $meta_key ] as $tab_key => $tab ) {
						$tabs[] = array(
							'default_label' => $tab['label'],
							'label'         => $tab['label'],
							'show_in_menu'  => $tab['show_in_menu'],
							'content'       => $tab['content'],
							'custom'        => $tab['custom'],
							'global'        => $tab['global'],
							'tab_key'       => $tab_key,
						);
					}

					$data[ $meta_key ] = $tabs;

				}
			}

			$trip_data[ $field_name ] = $data;

			return $trip_data;
		}

		/**
		 * Update Gallery data onto trip <meta class="">
		 *
		 * @since TODO: add when release
		 */
		public static function save_rest_data( $trip_data, $trip_id = 0 ) {
			$trip_data    = (array) $trip_data;
			$field_name   = 'utilities';
			$data         = isset( $trip_data[ $field_name ] ) ? $trip_data[ $field_name ] : array();
			$valid_fields = self::get_trip_meta_fields( $trip_id );

			foreach ( (array) $data as $name => $value ) {
				// error_log( $name );
				if ( array_key_exists( $name, $valid_fields ) ) {
					$value_to_update = gettype( $value ) === $valid_fields[ $name ]['type'] ? $value : $valid_fields[ $name ]['default'];

					switch ( $name ) {
						case 'wp_travel_trip_checkout_page_info': // cart & checkout > info content.
							$value_to_update = wp_kses_post( $value_to_update );
							break;
						case 'wp_travel_itinerary_custom_tab_cnt_':
							// Map Tab Data.
							$custom_tabs = array();
							foreach ( $value_to_update as  $trip_tab ) {

								if ( isset( $trip_tab['tab_key'] ) ) {
									$tab_key = $trip_tab['tab_key']; // quick fix.
								} else {
									$tab_key = $trip_tab['label']; // quick fix.
								}

								$custom_tabs[ $tab_key ]['label']         = $trip_tab['label'];
								$custom_tabs[ $tab_key ]['default_label'] = $trip_tab['label'];
								$custom_tabs[ $tab_key ]['content']       = wp_kses_post( $trip_tab['content'] );

								$custom_tabs[ $tab_key ]['show_in_menu'] = 'yes';
								$custom_tabs[ $tab_key ]['custom']       = 'yes';
								$custom_tabs[ $tab_key ]['global']       = 'yes';
							}
							$value_to_update = $custom_tabs;
							break;
						default:
							$value_to_update = sanitize_text_field( $value_to_update );
							break;
					}
				} else {
					continue;
				}

				update_post_meta( $trip_id, $name, $value_to_update );
			}
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_utilities'] = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_utilities';
			$field_label = __( 'WP Travel Utilities', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}

		public static function plugin_disabled_notice() {
			echo sprintf( '<p><strong>WP Travel Utilities</strong> is currently disabled please go to <a href="%s">Addons Settings</a> to enable it to get Custom Tabs, FAQs, Checkout options and more. </p>', esc_url( admin_url( 'edit.php?post_type=itinerary-booking&page=settings#wp-travel-tab-content-addons_settings' ) ) );
		}

		/**
		 * Utilities Frontend assets.
		 */
		public static function frontend_assets() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

			// Styles.
			// wp_register_style( 'wp-travel-utilities-front-end-css', self::$plugin_url . 'assets/css/wp-travel-utilities-front-end' . $suffix . '.css', array( 'wp-travel-slick' ), self::$version );
			wp_enqueue_style( 'wp-travel-utilities-front-end-css', self::$plugin_url . 'assets/css/wp-travel-utilities-front-end' . $suffix . '.css', array( 'wp-travel-slick' ), self::$version );

			// Scripts.
			// wp_register_script( 'wp-travel-utilities-front-end-js', self::$plugin_url . 'assets/js/wp-travel-utilities-front-end' . $suffix . '.js', array( 'jquery', 'wp-travel-slick', 'collapse-js' ), self::$version, true );
			wp_enqueue_script( 'wp-travel-utilities-front-end-js', self::$plugin_url . 'assets/js/wp-travel-utilities-front-end' . $suffix . '.js', array( 'jquery', 'wp-travel-slick', 'collapse-js' ), self::$version, true );
		}

		/**
		 * Utilities admin assets.
		 */
		public static function admin_assets() {

			$suffix         = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$screen         = get_current_screen();
			$allowed_screen = array( 'edit-itineraries', WP_TRAVEL_POST_TYPE, 'itineraries_page_settings', 'itinerary-booking', 'itinerary-booking_page_settings' );
			if ( in_array( $screen->id, $allowed_screen ) ) {
				// Styles.
				wp_enqueue_style( 'wp-travel-utilities-back-end-css', self::$plugin_url . 'assets/css/wp-travel-utilities-back-end' . $suffix . '.css', '', self::$version );

				// Scripts.
				wp_enqueue_script( 'wp-travel-utilities-back-end-js', self::$plugin_url . 'assets/js/wp-travel-utilities-back-end' . $suffix . '.js', array( 'jquery', 'jquery-ui-sortable' ), self::$version, true );
				wp_enqueue_script( 'wp-travel-utilities-custom-media', self::$plugin_url . 'assets/js/custom-media' . $suffix . '.js', array( 'jquery' ), self::$version, true );

				// React Stuffs. @since WP Travel v4.0.0
				$deps = include_once sprintf( '%sapp/build/wp-travel-utilities-admin-trip-options.asset.php', plugin_dir_path( __FILE__ ) );
				wp_enqueue_script( 'wp-travel-utilities-admin-trip-options', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-utilities-admin-trip-options.js', $deps['dependencies'], $deps['version'], true );
				wp_enqueue_style( 'wp-travel-utilities-admin-trip-options-style', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-utilities-admin-trip-options.css', array( 'wp-components' ), $deps['version'] );

			}

			if ( method_exists( 'WP_Travel', 'is_page' ) ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-utilities-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( 'wp-travel-wp-travel-utilities-settings', plugin_dir_url( __FILE__ ) . 'app/build/wp-travel-utilities-settings.js', $deps['dependencies'], $deps['version'], true );

				wp_localize_script( 'wp-travel-wp-travel-utilities-settings', 'WTTG', array(
					'wp_travel_version' => WP_TRAVEL_VERSION,
				));
			}
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			include sprintf( '%s/inc/class-wp-travel-utilities-shortcodes.php', self::$abspath );

			// Customized Trip Codes.
			include sprintf( '%s/inc/admin/class-wp-travel-utilities-customized-trip-code.php', self::$abspath );

			// Email Template Options.
			include sprintf( '%s/inc/class-wp-travel-utilities-email-options.php', self::$abspath );

			// Functions.
			include sprintf( '%s/inc/functions.php', self::$abspath );

			if ( is_admin() ) {
				include sprintf( '%s/inc/admin/wp-travel-utilities-additional-settings.php', self::$abspath );
			}
			include sprintf( '%s/inc/class-wp-travel-cart-checkout.php', self::$abspath );

			// trip time module
			include sprintf( '%s/inc/trip-time.php', self::$abspath );
		}

		/**
		 * Return True to allow custom tab when activate utilities.
		 *
		 * @param boolean $enable Boolean value.
		 * @return boolean
		 */
		public static function enable_custom_tab( $enable ) {
			return true;
		}

		/**
		 * Return Pricing option list.
		 *
		 * @param array $pricing_options_list Lists.
		 * @return array
		 */
		public static function custom_pricing_option( $pricing_options_list ) {
			$pricing_options_list['custom-booking'] = __( 'Custom Booking', 'wp-travel-pro' );
			return $pricing_options_list;
		}

		/**
		 * Return Pricing option list.
		 *
		 * @param array $pricing_options_list Lists.
		 * @return array
		 */
		// public static function custom_pricing_option_content( $post_id ) {
		// 	include sprintf( '%s/inc/custom-pricing-options.php', self::$abspath );
		// }

		/**
		 * Return Pricing option list.
		 *
		 * @param array $pricing_options_list Lists.
		 * @return array
		 */
		public static function custom_pricing_option_content_save( $trip_meta, $post_id ) {
			if ( isset( $_POST['wp_travel_custom_booking_type'] ) ) {
				$trip_meta['wp_travel_custom_booking_type'] = $_POST['wp_travel_custom_booking_type'];
			}
			if ( isset( $_POST['wp_travel_custom_booking_link'] ) ) {
				$trip_meta['wp_travel_custom_booking_link'] = $_POST['wp_travel_custom_booking_link'];
			}
			if ( isset( $_POST['wp_travel_custom_booking_link_text'] ) ) {
				$trip_meta['wp_travel_custom_booking_link_text'] = $_POST['wp_travel_custom_booking_link_text'];
			}
			$trip_meta['wp_travel_custom_booking_link_open_in_new_tab'] = 0;
			if ( isset( $_POST['wp_travel_custom_booking_link_open_in_new_tab'] ) ) {
				$trip_meta['wp_travel_custom_booking_link_open_in_new_tab'] = $_POST['wp_travel_custom_booking_link_open_in_new_tab'];
			}
			if ( isset( $_POST['wp_travel_custom_booking_form'] ) ) {
				$trip_meta['wp_travel_custom_booking_form'] = $_POST['wp_travel_custom_booking_form'];
			}

			$trip_meta['wp_travel_is_global_faq'] = isset( $_POST['wp_travel_is_global_faq'] ) ? ( wp_unslash( $_POST['wp_travel_is_global_faq'] ) ) : array();

			return $trip_meta;
		}

		/**
		 * Render HTML for Custom Pricing option.
		 *
		 * @param int $trip_id Current Trip id.
		 */
		// public static function content_pricing_option( $trip_id ) {
		// 	die;
		// 	if ( ! $trip_id ) {
		// 		return;
		// 	}

		// 	$custom_booking_type = get_post_meta( $trip_id, 'wp_travel_custom_booking_type', true );
		// 	if ( ! empty( $custom_booking_type ) ) {
		// 		ob_start();
		// 		echo '<div class="wp-travel-' . esc_attr( $custom_booking_type ) . '">';
		// 		switch ( $custom_booking_type ) {
		// 			case 'custom-link':
		// 				$link      = get_post_meta( $trip_id, 'wp_travel_custom_booking_link', true );
		// 				$link_text = get_post_meta( $trip_id, 'wp_travel_custom_booking_link_text', true );
		// 				$link_text = $link_text ? $link_text : __( 'Book Now', 'wp-travel-pro' );
		// 				$new_tab   = get_post_meta( $trip_id, 'wp_travel_custom_booking_link_open_in_new_tab', true );

		// 				$target = '';
		// 				if ( '1' == $new_tab ) {
		// 					$target = 'target="_blank"';
		// 				}
		// 				if ( $link ) {
		// 					echo '<a href="' . esc_url( $link ) . '" ' . esc_attr( $target ) . '>' . esc_html( $link_text ) . '</a>';
		// 				}
		// 				break;
		// 			case 'custom-form':
		// 				$html = get_post_meta( $trip_id, 'wp_travel_custom_booking_form', true );
		// 				echo apply_filters( 'the_content', $html );
		// 				break;
		// 		}
		// 		echo '</div>';
		// 	}
		// }
	}
endif;
