<?php
/**
 * WP Travel Wishlists Core Class.
 *
 * @package wp-travel-wishlists-core
 * @category Core
 * @author WEN Solutions
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP Travel Wishlists core.
if ( ! class_exists( 'WP_Travel_Wishlists_Core' ) ) :
	/**
	 * Core Class
	 */
	class WP_Travel_Wishlists_Core {

		const WP_TRAVEL_WISHLISTS_HANDLE = 'wp_travel_wishlists_';
		/**
		 * ABSPATH
		 *
		 * @var string $abspath
		 */
		protected static $abspath;

		/**
		 * Plugin File Path
		 *
		 * @var string $plugin_file
		 */
		protected static $plugin_file;

		/**
		 * Plugin File URL
		 *
		 * @var string $plugin_url
		 */
		protected static $plugin_url;

		/**
		 * Plugin Version
		 *
		 * @var string $version
		 */
		protected static $version;

		/**
		 * The single instance of the class.
		 *
		 * @var WP Travel Wishlists Core
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * Main WP_Travel_Wishlists_Core Instance.
		 * Ensures only one instance of WP_Travel_Wishlists_Core is loaded or can be loaded.
		 *
		 * @since 1.0.0
		 * @static
		 * @see WP_Travel_Wishlists_Core()
		 * @return WP_Travel_Wishlists_Core - Main instance.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Init core.
		 *
		 * @param array $plugin_data Plagin data.
		 */
		public static function init( $plugin_data ) {
			self::$abspath     = dirname( __FILE__ ) . '/';
			self::$plugin_file = __FILE__;
			self::$plugin_url  = plugin_dir_url( __FILE__ );
			self::$version     = $plugin_data['version'];

			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$enable_wishlists = isset( $settings['show_wp_travel_wishlists'] ) ? $settings['show_wp_travel_wishlists'] : 'yes';

			add_filter( 'wp_travel_settings_fields', array( 'WP_Travel_Wishlists_Core', 'settings_fields' ) );
			add_action( 'wp_travel_addons_setings_tab_fields', array( 'WP_Travel_Wishlists_Core', 'plugin_action' ) );
			add_action( 'admin_enqueue_scripts', array( 'WP_Travel_Wishlists_Core', 'admin_assets' ) );
			// if ( 'yes' !== $enable_wishlists ) {
			// 	return;
			// }
			self::includes();

			add_action( 'wp_enqueue_scripts', array( 'WP_Travel_Wishlists_Core', 'frontend_assets' ) );

			/**
			 * Filter 
			 * 
			 * @since 4.0.6
			 */
			add_filter( 'wp-travel-script-handles', function( $handles ) {
				$handles[] = 'wp-travel-list-wishlists';
				$handles[] = 'wp-travel-wishlists';
				return $handles;
			} );
		}

		/**
		 * Default settings fields.
		 *
		 * @param array $settings WP Travel Settings.
		 * @return array
		 */
		public static function settings_fields( $settings ) {
			$settings['show_wp_travel_wishlists'] = 'yes';
			return $settings;
		}

		/**
		 * Plugin action to show / hide plugin settings and features.
		 */
		public static function plugin_action() {
			if ( function_exists( 'wptravel_get_settings' ) ) {
				$settings = wptravel_get_settings();
			} else {
				$settings = wp_travel_get_settings();
			}

			$field_name  = 'show_wp_travel_wishlists';
			$field_label = __( 'WP Travel Wishlists', 'wp-travel-pro' );

			$field_value = isset( $settings[ $field_name ] ) ? $settings[ $field_name ] : 'yes'; ?>
			<table class="form-table">
				<tr>
					<th>
						<label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label>
					</th>
					<td>
						<span class="show-in-frontend checkbox-default-design">
							<label data-on="ON" data-off="OFF">
								<input value="no" name="<?php echo esc_attr( $field_name ); ?>" type="hidden" />
								<input <?php checked( $field_value, 'yes' ); ?> value="yes" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" type="checkbox" />
								<span class="switch"></span>
							</label>
						</span>
						<p class="description"><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html__( 'Show all your "', 'wp-travel-pro' ) . esc_html( $field_label ) . esc_html__( '" settings and enable its feature.', 'wp-travel-pro' ); ?></label></p>
					</td>
				</tr>
			</table>
			<?php

		}


		/**
		 * Frontend assets.
		 */
		public static function frontend_assets() {

			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			// Styles
			wp_register_style( 'wp-travel-list-wishlists', self::$plugin_url . 'assets/css/wp-travel-wishlists' . $suffix . '.css' );
			wp_enqueue_style( 'wp-travel-list-wishlists' );

			// Scripts.
			wp_register_script( 'wp-travel-list-wishlists', self::$plugin_url . 'assets/js/dist/list-wishlists' . $suffix . '.js', array( 'jquery' ), self::$version, true );
			wp_register_script( 'wp-travel-wishlists', self::$plugin_url . 'assets/js/dist/wishlists' . $suffix . '.js', array( 'jquery' ), self::$version, true );

			// Enqueued script.
			wp_enqueue_script( 'wp-travel-wishlists' );

		}

		/**
		 * Admin assets.
		 */
		public static function admin_assets() {

			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			$screen = get_current_screen();
			if (  method_exists( 'WP_Travel',  'is_page' )  ) { // @since WP Travel 4.4.2
				$is_settings_page = WP_Travel::is_page( 'settings', true );
			} else {
				$is_settings_page = 'itinerary-booking_page_settings' == $screen->id;
			}
			if ( $is_settings_page ) {
				$deps                   = include_once sprintf( '%s/app/build/wp-travel-settings.asset.php', plugin_dir_path( __FILE__ ) );
				$deps['dependencies'][] = 'jquery';
				wp_enqueue_script( self::WP_TRAVEL_WISHLISTS_HANDLE . 'admin-settings', plugin_dir_url( __FILE__ ) . '/app/build/wp-travel-settings.js', $deps['dependencies'], $deps['version'], true );
			}
		}

		/**
		 * Include required core files used in admin and on the frontend.
		 *
		 * @return void
		 */
		public static function includes() {

			if ( ! class_exists( 'WP_Travel' ) ) {
				return;
			}
			/**
			 * Include required files
			 */
			include sprintf( '%s/inc/functions.php', self::$abspath );

		}

		/**
		 * What type of request is this?
		 *
		 * @param  string $type admin, ajax, cron or frontend.
		 * @return bool
		 */
		private static function is_request( $type ) {
			switch ( $type ) {
				case 'admin':
					return is_admin();
				case 'ajax':
					return defined( 'DOING_AJAX' );
				case 'cron':
					return defined( 'DOING_CRON' );
				case 'frontend':
					return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
			}
		}


	}
endif;
